<?php

namespace Txd\Core;

use Illuminate\Contracts\Routing\UrlRoutable;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Routing\Exceptions\BackedEnumCaseNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Reflector;
use Illuminate\Support\Str;

class ImplicitRouteBinding extends \Illuminate\Routing\ImplicitRouteBinding
{
    /**
     * Resolve the implicit route bindings for the given route.
     *
     * @param  \Illuminate\Container\Container  $container
     * @param  \Illuminate\Routing\Route  $route
     * @return void
     *
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException<\Illuminate\Database\Eloquent\Model>
     * @throws \Illuminate\Routing\Exceptions\BackedEnumCaseNotFoundException
     */
    public static function resolveForRoute($container, $route)
    {
        $parameters = $route->parameters();
        $controllerMapping = new Collection;

        if ($route->controller && method_exists(get_class($route->controller), 'getParametersMapping')) {
            /** @var Collection */
            $controllerMapping = optional($route->controller)->getParametersMapping() ?? new Collection;
        }

        $route = static::resolveBackedEnumsForRoute($route, $parameters, $controllerMapping);

        $routeParameters = [];
        foreach ($parameters as $key => $value) {
            $obj = new class
            {
                public $name;

                public function getName()
                {
                    return $this->name;
                }

                public function getType()
                {
                    return null;
                }
            };
            $obj->name = $key;
            $routeParameters[] = $obj;
        }
        $signatureParameters = $route->signatureParameters(['subClass' => UrlRoutable::class]);
        $mappedParameters = array_filter($route->signatureParameters(), fn ($parameter) => $controllerMapping->has($parameter->getName()));

        $mergedParameters = [];
        foreach ($signatureParameters as $key => $value) {
            $mergedParameters[$key] = $value;
        }
        foreach ($mappedParameters as $key => $value) {
            $mergedParameters[$key] = $value;
        }

        ksort($mergedParameters);

        $mergedParameters = array_merge($routeParameters, $mergedParameters);

        foreach ($mergedParameters as $parameter) {
            if (!$parameterName = static::getParameterName($parameter->getName(), $parameters, $controllerMapping)) {
                if ($controllerMapping->has($parameter->getName())) {
                    $modelName = $controllerMapping[$parameter->getName()]->parameterClass;
                    $route->setParameter($parameter->getName(), new $modelName);
                }

                continue;
            }

            $parameterValue = $parameters[$parameterName];

            if ($parameterValue instanceof UrlRoutable) {
                continue;
            }
            $className = Reflector::getParameterClassName($parameter);
            if (is_null($className)) {
                $className = $controllerMapping[$parameter->getName()]->parameterClass ?? null;
            }
            if (is_null($className)) {
                continue;
            }
            $instance = $container->make($className);

            $parent = $route->parentOfParameter($parameterName);

            $routeBindingMethod = $route->allowsTrashedBindings() && in_array(SoftDeletes::class, class_uses_recursive($instance))
                        ? 'resolveSoftDeletableRouteBinding'
                        : 'resolveRouteBinding';

            if ($parent instanceof UrlRoutable && ($route->enforcesScopedBindings() || array_key_exists($parameterName, $route->bindingFields()))) {
                $childRouteBindingMethod = $route->allowsTrashedBindings() && in_array(SoftDeletes::class, class_uses_recursive($instance))
                            ? 'resolveSoftDeletableChildRouteBinding'
                            : 'resolveChildRouteBinding';

                if (!$model = $parent->{$childRouteBindingMethod}(
                    $parameterName, $parameterValue, $route->bindingFieldFor($parameterName)
                )) {
                    throw (new ModelNotFoundException)->setModel(get_class($instance), [$parameterValue]);
                }
            } elseif (!$model = $instance->{$routeBindingMethod}($parameterValue, $route->bindingFieldFor($parameterName))) {
                throw (new ModelNotFoundException)->setModel(get_class($instance), [$parameterValue]);
            }

            $route->setParameter($parameterName, $model);
        }
    }

    /**
     * Resolve the Backed Enums route bindings for the route.
     *
     * @param  \Illuminate\Routing\Route  $route
     * @param  array  $parameters
     * @return \Illuminate\Routing\Route
     *
     * @throws \Illuminate\Routing\Exceptions\BackedEnumCaseNotFoundException
     */
    protected static function resolveBackedEnumsForRoute($route, $parameters, ?Collection $mappings = null)
    {
        foreach ($route->signatureParameters(['backedEnum' => true]) as $parameter) {
            if (!$parameterName = static::getParameterName($parameter->getName(), $parameters, $mappings)) {
                continue;
            }

            $parameterValue = $parameters[$parameterName];

            $backedEnumClass = (string) $parameter->getType();

            $backedEnum = $backedEnumClass::tryFrom((string) $parameterValue);

            if (is_null($backedEnum)) {
                throw new BackedEnumCaseNotFoundException($backedEnumClass, $parameterValue);
            }

            $route->setParameter($parameterName, $backedEnum);
        }

        return $route;
    }

    /**
     * Return the parameter name if it exists in the given parameters.
     *
     * @param  string  $name
     * @param  array  $parameters
     * @return string|null
     */
    protected static function getParameterName($name, $parameters, ?Collection $mappings = null)
    {
        if (array_key_exists($name, $parameters)) {
            return $name;
        }

        if (array_key_exists($snakedName = Str::snake($name), $parameters)) {
            return $snakedName;
        }
        if (!is_null($mappings) && !$mappings->isEmpty()) {
            $name = optional($mappings[$name] ?? null)->parameterName;
            if (array_key_exists($name, $parameters)) {
                return $name;
            }
        }
    }
}
