<?php

namespace JSM\ExecutionTimeMeasure;

use JSM\ExecutionTimeMeasure\Exception\ConnectionException;
use JSM\ExecutionTimeMeasure\Exception\ConfigurationException;

/**
 * ExecutionTimeMeasure Client Class
 *
 * @author Fabio Spadea <fabio@techseed.it>
 */
class ETMClient
{

    /**
     * Instance instances array
     * @var array
     */
    protected static $instances = array();


    /**
     * Instance ID
     * @var string
     */
    protected $instance_id;


    /**
     * Server Host
     * @var string
     */
    protected $host = '127.0.0.1';


    /**
     * Server Port
     * @var integer
     */
    protected $port = 8125;


    /**
     * Last message sent to the server
     * @var string
     */
    protected $message = '';


    /**
     * Timeout for creating the socket connection
     * @var null|float
     */
    protected $timeout;

    /**
     * Whether or not an exception should be thrown on failed connections
     * @var bool
     */
    protected $throwConnectionExceptions = true;

    /**
     * Record metric start time
     * @var array
     */
    protected $metricTiming;

    /**
     * Socket pointer for sending metrics
     * @var resource
     */
    protected $socket;

    /**
     * Generic tags
     * @var array
     */
    protected $tags = [];

    protected $middleware_fired = false;

    /**
     * Singleton Reference
     * @param  string $name Instance name
     * @return ETMClient ETMClient instance
     */
    public static function instance($name = 'default')
    {
        if (! isset(self::$instances[$name])) {
            self::$instances[$name] = new static($name);
        }
        return self::$instances[$name];
    }


    /**
     * Create a new instance
     * @param string $instance_id
     * @return void
     */
    public function __construct($instance_id = null)
    {
        $this->instance_id = $instance_id ?: uniqid();

        if (empty($this->timeout)) {
            $this->timeout = ini_get('default_socket_timeout');
        }
    }

    /**
     * Get string value of instance
     * @return string String representation of this instance
     */
    public function __toString()
    {
        return 'ExecutionTimeMeasure\ETMClient::[' . $this->instance_id . ']';
    }


    /**
     * Initialize Connection Details
     * @param array $options Configuration options
     * @return ETMClient This instance
     * @throws ConfigurationException If port is invalid
     */
    public function configure(array $options = array())
    {
        if (isset($options['host'])) {
            $this->host = $options['host'];
        }
        if (isset($options['port'])) {
            if (!is_numeric($options['port']) || is_float($options['port']) || $options['port'] < 0 || $options['port'] > 65535) {
                throw new ConfigurationException($this, 'Port is out of range');
            }
            $this->port = $options['port'];
        }

        if (isset($options['timeout'])) {
            $this->timeout = $options['timeout'];
        }

        if (isset($options['throwConnectionExceptions'])) {
            $this->throwConnectionExceptions = $options['throwConnectionExceptions'];
        }

        if (isset($options['tags'])) {
            $this->tags = $options['tags'];
        }

        return $this;
    }


    /**
     * Get Host
     * @return string Host
     */
    public function getHost()
    {
        return $this->host;
    }


    /**
     * Get Port
     * @return string Port
     */
    public function getPort()
    {
        return $this->port;
    }


    /**
     * Get Last Message
     * @return string Last message sent to server
     */
    public function getLastMessage()
    {
        return $this->message;
    }

    public function middlewareFire(){
        $this->middleware_fired = true;
    }

    public function isMiddlewareFired(){
        return $this->middleware_fired;
    }

    /**
     * Start timing the given metric
     * @param  string $metric Metric to time
     * @return $this
     */
    public function startTiming($metric, array $tags = array())
    {
        $this->metricTiming[$metric]["start"] = microtime(true);
        // $this->metricTiming[$metric]["tags"] = $tags;
        return $this;
    }

    /**
     * End timing the given metric and record
     * @param  string $metric Metric to time
     * @param  array $tags A list of metric tags values
     * @return $this
     * @throws ConnectionException
     */
    public function endTiming($metric, array $tags = array()){
        
        $timer_start = $this->metricTiming[$metric]["start"] ?? 0;
        $timer_end = microtime(true);
        // $time = round(($timer_end - $timer_start) * 1000, 4); //ms
        $time = $timer_end - $timer_start; //sec
        
        $this->metricTiming[$metric]["end"] = $timer_end;
        $this->metricTiming[$metric]["execution_time"] = $time;

        // $existing_tags = isset($this->metricTiming[$metric]["tags"]) && is_array($this->metricTiming[$metric]["tags"]) ? $this->metricTiming[$metric]["tags"] : [];
        // $this->metricTiming[$metric]["tags"] = array_merge($existing_tags, $tags);

        return $this;
    }


    /**
     * @throws ConnectionException
     * @return resource
     */
    protected function getSocket()
    {
        if (!$this->socket) {
            $this->socket = @fsockopen('udp://' . $this->host, $this->port, $errno, $errstr, $this->timeout);
            if (!$this->socket) {
                throw new ConnectionException($this, '(' . $errno . ') ' . $errstr);
            }
        }

        return $this->socket;
    }


    /**
     * Send Data to ETM Server
     * @param  bool $log_to_file se impostato a TRUE salva le info anche in un file di log specifico
     * @return $this
     * @throws ConnectionException If there is a connection problem with the host
     */
    public function send($log_to_file = false){

        if(config("jsm_execution_time_measure.enabled") == false){
            return $this;
        }

        $message_info = [];

        $message_info["date_time"] = date("Y-m-d H:i:s");
        $message_info["url"] = request()->fullUrl();
        $message_info["ip"] = request()->ip();
        $message_info["metrics"] = $this->metricTiming;
        $message_info["app_name"] = config("app.name");

        $this->message = json_encode($message_info);

        try {

            $socket = $this->getSocket();
            
            @fwrite($socket, $this->message);
            fflush($socket);

            if($log_to_file){
                $filename = date("Y-m-d").'_jsm_etm' . '.log';
                \File::append( storage_path('logs' . DIRECTORY_SEPARATOR . $filename), $this->message."\n");
            }
            
        } catch (ConnectionException $e) {
            if ($this->throwConnectionExceptions) {
                throw $e;
            } else {
                trigger_error(
                    sprintf('ETM server connection failed (udp://%s:%d)', $this->host, $this->port),
                    E_USER_WARNING
                );
            }
        }

        return $this;
    }

}
