<?php

namespace JSM\Postman;

use Illuminate\Support\Facades\Log;

class SMSMessage
{

    private $sender = null;
    private $recipient = null;
    private $message = null;

    public function __construct()
    {

    }

    /**
     * Set the sender for the SMS message. The string has to be between 3 and 11 characters long
     *
     * @param string $sender
     *
     * @return self
     */
    public function sender(string $sender)
    {
        $count = strlen($sender);
        if ($count >= 3 && $count <= 11) {
            $this->sender = $sender;
        } else {
            throw new \InvalidArgumentException("Sender must be between 3 and 11 characters long");
        }
        return $this;
    }

    /**
     * set the recipient number. checks for E.164 international format
     *
     * @param string $recipient
     *
     * @return self
     */
    public function recipient(string $recipient)
    {
        if (class_exists("\libphonenumber\PhoneNumberUtil")) {
            $phoneUtil = \libphonenumber\PhoneNumberUtil::getInstance();
            $formatted = $phoneUtil->parse($recipient, \libphonenumber\PhoneNumberFormat::E164);
            if ($phoneUtil->isValidNumber($formatted)) {
                $recipient = $phoneUtil->format($formatted, \libphonenumber\PhoneNumberFormat::E164);
            } else {
                throw new \InvalidArgumentException("Phone number is invalid. Please use E.164 format (International Prefix and without spaces or dots)");
            }
        }

        $this->recipient = $recipient;

        return $this;
    }

    /**
	 * set the message string
     * @param string $message
     * 
     * @return self
     */
    public function message(string $message)
    {
        if (strlen($message) > 0) {
            $this->message = $message;
        }
        return $this;
    }

    /**
	 * send the sms message
	 * 
     * @return bool
     */
    public function send()
    {
        $esito = false;
        if (!is_null($this->sender) && !is_null($this->recipient) && !is_null($this->message)) {
            $json_message = array(
                // Source is required
                'Source' => $this->sender,
                // Destination is required
                'Destination' => $this->recipient,
                // Message is required
                'Message' => $this->message);

            try {
                $cipher = new AESCipher(config("jsm_postman.postman_key_secret"));
                $messaggio = $cipher->encrypt(json_encode($json_message));

                $url = config("jsm_postman.sms_endpoint", 'https://postman.sinergo.it/api/v3/postman/sms');

                $eol = "\r\n";
                $data = '';
                $mime_boundary = md5(time());
                //
                $data .= '--' . $mime_boundary . $eol;
                $data .= 'Content-Disposition: form-data; name="id"' . $eol . $eol;
                $data .= config("jsm_postman.postman_key_id") . $eol;
                $data .= '--' . $mime_boundary . $eol;
                $data .= 'Content-Disposition: form-data; name="info"' . $eol . $eol;
                $data .= $messaggio . $eol;
                $data .= '--' . $mime_boundary . $eol;
                $data .= "--" . $mime_boundary . "--" . $eol . $eol;

                // use key 'http' even if you send the request to https://...
                $options = array(
                    'http' => array(
                        'header' => "Content-type: multipart/form-data; boundary=" . $mime_boundary,
                        'method' => 'POST',
                        'content' => $data,
                    ),
                );

                $context = stream_context_create($options);
                $result = file_get_contents($url, FILE_TEXT, $context);
                $result = json_decode($result, true);
            } catch (\Exception $e) {
                $result = [
                    "status" => "KO",
                    "message" => $e->getMessage(),
                ];
            }

            if (!is_array($result)) {
                $esito = false;
            } else {
                if (array_key_exists("status", $result)) {
                    if ($result['status'] === "OK") {
                        $esito = true;
                    }
                }
            }

            if (!$esito) {
                \Log::error(get_called_class() . " | " . __FUNCTION__ . " | " . json_encode($result));

                if (config("app.debug")) {
                    throw new \JSM\Postman\Exceptions\PostmanException(json_encode($result));
                }
            }
        }

        return $esito;
    }

    public static function composeAndSend(string $sender, string $recipient, string $message){

        $esito = false;
        try{
            
            $sms = new \JSM\Postman\SMSMessage;
            $esito = $sms->sender($sender)->recipient($recipient)->message($message)->send();
        }catch(\Exception $e){
            
            \Log::error( __FUNCTION__ . " | " . $e->getMessage());
            if (config("app.debug")) {
                throw $e;
            }
            
        }
        
        return $esito;
    
    }

}
