<?php

namespace JSM\Postman;

use GuzzleHttp\Client;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\SentMessage;
use Symfony\Component\Mailer\Transport\AbstractTransport;

class PostmanTransport extends AbstractTransport {

	protected $api_endpoint;
	protected $api_key_id;
	protected $api_key_secret;

	public function __construct($api_endpoint, $api_key_id, $api_key_secret,EventDispatcherInterface $dispatcher = null, LoggerInterface $logger = null){

        parent::__construct($dispatcher,$logger);
		$this->api_endpoint = $api_endpoint;
		$this->api_key_id = $api_key_id;
		$this->api_key_secret = $api_key_secret;
	}

	public function __toString(): string
	{
		return "postman";
	}

	protected function doSend(SentMessage $message): void
	{
		/** @var \Symfony\Component\Mime\Email */
		$original = $message->getOriginalMessage();
		$envelope = $message->getEnvelope();
		$sender = $envelope->getSender()->toString();


        if($original->custom_attributes["save_source_on_disk"]??false){
            $this->saveSourceOnDisk($original);
        }

		$formatted_to = array_map(fn($rec)=>$rec->toString(), $original->getTo());
		$formatted_cc = array_map(fn($rec)=>$rec->toString(), $original->getCc());
		$formatted_bcc = array_map(fn($rec)=>$rec->toString(), $original->getBcc());

		$json_message = array(
			// Source is required
			'Source' => $sender,
			// Destination is required
			'Destination' => array(
				'ToAddresses' => $formatted_to,
				'CcAddresses' => $formatted_cc,
				'BccAddresses' => $formatted_bcc,
			),
			//Message is required
			'Message' => array(
				// Subject is required
				'Subject' => array(
					// Data is required
					'Data' => $original->getSubject(),
					'Charset' => 'UTF-8',
				),
				// Body is required
				'Body' => array(
					'Text' => array(
						// Data is required
						'Data' => $original->getTextBody(),
						'Charset' => $original->getTextCharset(),
					),
					'Html' => array(
						// Data is required
						'Data' => $original->getHtmlBody(),
						'Charset' => $original->getHtmlCharset(),
					),
				)
			)
		);

		if(count($original->getAttachments())>0 || ($original->custom_attributes["send_as_raw"]??false)){
			$raw_recipients = "";
			if(count($formatted_to)>0) $raw_recipients .= "To: ".implode(", ",$formatted_to)."\r\n";
			if(count($formatted_cc)>0) $raw_recipients .= "Cc: ".implode(", ",$formatted_cc)."\r\n";
			if(count($formatted_bcc)>0) $raw_recipients .= "Bcc: ".implode(", ",$formatted_bcc)."\r\n";
			$raw_recipients = "";
			$json_message["Message"]["Raw"] = $raw_recipients.$original->toString();
			$json_message["Message"]["MessageLogOverride"] = $original->custom_attributes["override_message"]??"false";
		}

		$cipher = new AESCipher($this->api_key_secret);
		$messaggio = $cipher->encrypt(json_encode($json_message));

		$url = $this->api_endpoint;


		$client = new Client();
		try {
			$client->post($url,['form_params' => [
				'id' => $this->api_key_id,
				'info' => $messaggio,
			]]);
		} catch (\GuzzleHttp\Exception\BadResponseException $ex) {
			throw new \JSM\Postman\Exceptions\PostmanException($ex->getResponse()->getBody()->getContents());
		}
	}

    protected function saveSourceOnDisk(\Symfony\Component\Mime\Email $message){
        $disk_name = $message->custom_attributes["disk_name"] ?? "";
        $path_prefix = $message->custom_attributes["path_prefix"] ?? "";
        $override_message_with_path = $message->custom_attributes["override_message_with_path"] ?? false;

        if(strlen($disk_name) > 0 && config("filesystems.disks.".$disk_name) !== null){

            $string_message = $message->toString();
            $file_name = date("Y-m-d_His")."_".\Illuminate\Support\Str::slug(microtime(true))."_".\Illuminate\Support\Str::slug($message->getSubject())."_".hash("sha256", $string_message).".eml";

            \Storage::Disk($disk_name)->put($path_prefix.$file_name, $string_message);

            //se questo parametro e' impostato a TRUE, andiamo a sovrascrivere l'attributo "override_message" eventualmente impostato nel metodo sendAsRaw() con il nome del file salvato
            if($override_message_with_path){
                $message->custom_attributes["override_message"] = $path_prefix.$file_name;
            }
        }
    }

}
