<?php

namespace JSM\Postman;

use GuzzleHttp\Client;

class SMSMessage
{
    private $sender = null;

    private $recipient = null;

    private $message = null;

    public function __construct() {}

    /**
     * Set the sender for the SMS message. The string has to be between 3 and 11 characters long
     *
     *
     * @return self
     */
    public function sender(string $sender)
    {
        $count = strlen($sender);
        if ($count >= 3 && $count <= 11) {
            $this->sender = $sender;
        } else {
            throw new \InvalidArgumentException('Sender must be between 3 and 11 characters long');
        }

        return $this;
    }

    /**
     * set the recipient number. checks for E.164 international format
     *
     *
     * @return self
     */
    public function recipient(string $recipient)
    {
        if (class_exists("\libphonenumber\PhoneNumberUtil")) {
            $phoneUtil = \libphonenumber\PhoneNumberUtil::getInstance();
            $formatted = $phoneUtil->parse($recipient, \libphonenumber\PhoneNumberFormat::E164);
            if ($phoneUtil->isValidNumber($formatted)) {
                $recipient = $phoneUtil->format($formatted, \libphonenumber\PhoneNumberFormat::E164);
            } else {
                throw new \InvalidArgumentException('Phone number is invalid. Please use E.164 format (International Prefix and without spaces or dots)');
            }
        }

        $this->recipient = $recipient;

        return $this;
    }

    /**
     * set the message string
     *
     * @return self
     */
    public function message(string $message)
    {
        if (strlen($message) > 0) {
            $this->message = $message;
        }

        return $this;
    }

    /**
     * send the sms message
     */
    public function send()
    {

        if (!is_null($this->sender) && !is_null($this->recipient) && !is_null($this->message)) {
            $json_message = [
                // Source is required
                'Source' => $this->sender,
                // Destination is required
                'Destination' => $this->recipient,
                // Message is required
                'Message' => $this->message];

            $cipher = new AESCipher(config('services.postman.secret'));
            $messaggio = $cipher->encrypt(json_encode($json_message));

            $url = config('services.postman.sms_endpoint', 'https://postman.sinergo.it/api/v3/postman/sms');

            $client = new Client;
            try {
                $client->post($url, ['form_params' => [
                    'id' => config('services.postman.key'),
                    'info' => $messaggio,
                ]]);
            } catch (\GuzzleHttp\Exception\BadResponseException $ex) {
                throw new \JSM\Postman\Exceptions\PostmanException($ex->getResponse()->getBody()->getContents());
            }
        }
    }

    public static function composeAndSend(string $sender, string $recipient, string $message)
    {

        $esito = false;
        try {

            $sms = new \JSM\Postman\SMSMessage;
            $esito = $sms->sender($sender)->recipient($recipient)->message($message)->send();
        } catch (\Exception $e) {

            \Log::error(__FUNCTION__.' | '.$e->getMessage());
            if (config('app.debug')) {
                throw $e;
            }

        }

        return $esito;

    }
}
