<?php

/**
 * Mettiamo qui una serie di wrapper e helper per interagire con lo storage S3 senza rifare tutto ogni volta
 *
 * @author fabio
 */
class AWS_S3{
	
	protected $_version = "0.1.2";
	protected $_changelog = "
		0.1.2	--	aggiunta info uploaded_file_name per recuperare il nome del file caricato; 
					metodo upload_file() con ritorno dell'esito
		0.1.1	--	Prima versione sotto package con controllo accesso alla prima escuzione del costruttore
	";


	protected $s3Adapter;
	protected $s3Client;
	protected $disk_name = "";


	protected $default_dir = "";
	protected $StorageClass = "STANDARD"; //STANDARD | REDUCED_REDUNDANCY
	protected $visibility = "public";

	protected $documentazione_attiva = false;
	
	protected $uploaded_file_name = null;

	/**
	 * 
	 * @param string $disk_name e' il nome del disco definito nel file config/filesystems
	 */
	public function __construct($disk_name){
		
		/*
		 * log documentazione
		 */
		if(class_exists("jsmDoc")){

			try{
				\jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__);
				$this->documentazione_attiva = true;
			}catch(\Exception $ex){
				\Log::error("Errore creazione documentazione: ".$ex->getMessage());

				if(config('app.debug')){
					throw $ex;
				}
			}
		}
		
		$this->disk_name = $disk_name;
		$this->s3Adapter = \Storage::disk($this->disk_name)->getAdapter();
		$this->s3Client = $this->s3Adapter->getClient();
		
		/*
		 * questo ci serve non tanto per verificare la regione di un bucket ordinario, ma per "testare" una chiamata in modo da evidenziare eventuali errori 
		 * (ad esempio se mancano i certficati per curl)
		 * 
		 * se qualcosa non funzionasse, dovrebbe generarsi un'eccezione
		 * 
		 * usiamo la sessione in modo che facciamo il controllo solo la prima volta per ogni disco
		 */
		if(\Session::get($disk_name."_S3_access") === null){
			$this->s3Client->determineBucketRegion("jsm");
			\Session::put([$disk_name."_S3_access" => true]);
			\Session::save();
		}
	}
	
	
	public function get_uploaded_file_name() {
		return $this->uploaded_file_name;
	}

		
	public function set_default_dir($dirname){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if(!is_string($dirname)){ $dirname = ""; }
		$this->default_dir = $dirname;
	}
	
	public function setDefaultStorageClass($StorageClass){
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		$this->StorageClass = $StorageClass;
	}

	public function setDefaultVisibility($visibility){
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		$this->visibility = $visibility;
	}

	

	/**
	 * verifica se l'oggetto passato (cartella o file) esiste
	 * 
	 * @param string $name
	 * @param mixed $dir se e' NULL viene usata la cartella di default come base di partenza
	 * @return boolean
	 */
	public function doesObjectExist($name, $dir=null){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		$object_path = add_ending_slash($dir).$name;
		
		return $this->s3Client->doesObjectExist($this->s3Adapter->getBucket(), $object_path);
	}


	/**
	 * verifica che l'elemento passato come name sia una cartella esistente
	 * 
	 * @param string $name
	 * @param mixed $dir se e' NULL viene presa la cartella di default come base di partenza
	 * @return boolean
	 */
	public function isDir($name, $dir=null){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		if(!$this->doesObjectExist($name, $dir)){
			return false;
		}
		
		$item = $this->getObject($name, $dir);
		
		return (isset($item["ContentType"]) && $item["ContentType"] == "application/x-directory");
	}
	
	
	/**
	 * verifica che l'elemento passato come name sia un file esistente
	 * 
	 * @param string $name
	 * @param mixed $dir se e' NULL viene presa la cartella di default come base di partenza
	 * @return boolean
	 */
	public function isFile($name, $dir=null){
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if(!$this->doesObjectExist($name, $dir)){
			return false;
		}
		
		return !$this->isDir($name, $dir);
	}

	
	/**
	 * ritorna il content-type dell'elemento passato (se esiste), NULL altrimenti
	 * 
	 * @param type $name
	 * @param type $dir
	 * @return type
	 */
	public function contentType($name, $dir=null){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		if(!$this->doesObjectExist($name, $dir)){
			return null;
		}
		
		$item = $this->getObject($name, $dir);
		
		return $item["ContentType"] ?? "";
	}


	/**
	 * ritorna l'url diretto al file su S3
	 * <br>se il file e' visibile pubblicamente sara' sufficiente visitarlo per scaricarlo
	 * 
	 * @param type $name
	 * @param type $dir
	 * @return type
	 */
	public function getPublicURL($name, $dir=null){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		$object_path = add_ending_slash($dir).$name;
		
		if(!$this->doesObjectExist($name, $dir)){
			return null;
		}
		
		return \Storage::disk($this->disk_name)->url($object_path);
	}
	
	
	/**
	 * recupera un oggetto da S3
	 * 
	 * @param type $name
	 * @param type $dir
	 * @param type $options
	 * @return type
	 */
	public function getObject($name, $dir=null, $options = []){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if(!is_array($options)){ $options = []; }
		
		$options["Bucket"] = $this->s3Adapter->getBucket();
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		$object_path = add_ending_slash($dir).$name;
		
		$options["Key"] = $object_path;
		
		if(!$this->doesObjectExist($name, $dir)){
			return null;
		}
		
		return $this->s3Client->getObject($options);
	}
	
	
	/**
	 * recupera il contenuto binario di un file su S3 e ne ritorna lo stream per essere mandato al browser
	 * 
	 * @param type $name
	 * @param type $dir
	 * @param type $stream_filename
	 * @return type
	 */
	public function getResponse($name, $dir=null, $stream_filename = null){

		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		$stream = $this->getObject($name, $dir);
		
		if($stream === null){
			return response("File Not Found!", 404);
		}
		
		return response($stream['Body'], 200)->withHeaders([
			'Content-Type'        => $this->contentType($name, $dir),
			'Content-Length'      => $stream['ContentLength'],
			'Content-Disposition' => 'inline; filename="' . ($stream_filename ?? $name) . '"'
		]);
	}
	
	
	
	/**
	 * permette di salvare un file su S3
	 * 
	 * @param binary $content il contenuto binario del file da caricare (ottenuto ad esempio con file_get_contents(file))
	 * @param string $name nome di destinazione del file
	 * @param mixed $dir nome della cartella in cui salvare il file (se e' NULL viene usata la cartella di default)
	 * @param array $options eventuali opzioni ad esempio storageClass o visiblity
	 * @param boolean $escape_file_name [default: TRUE] se impostato a TRUE effettua un escape del nome del file con la funzione str_slug($nome, "_")
	 * @return boolean
	 */
	public function upload_file($content, $name, $dir=null, $options = [], $escape_file_name = true){
		
		if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		if($dir === null){
			$dir = $this->default_dir;
		}
		
		if($escape_file_name){
			$basename = pathinfo($name, PATHINFO_FILENAME);
			$ext = pathinfo($name, PATHINFO_EXTENSION);
			$name = str_slug($basename, "_").".".$ext;
		}
		
		$this->uploaded_file_name = $name; //in questo modo potremo recuperare il vero nome del file caricato
		$object_path = add_ending_slash($dir).$name;
		
		return \Storage::disk($this->disk_name)->put($object_path, $content, $options);
	}
}
