<?php

/**
 * Created by Cristian.
 * Date: 19/09/16 11:58 PM.
 */

namespace Reliese\Coders\Model;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Reliese\Meta\Blueprint;
use Reliese\Support\Classify;
use Reliese\Meta\SchemaManager;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Database\DatabaseManager;

class Factory
{
    /**
     * @var \Illuminate\Database\DatabaseManager
     */
    private $db;

    /**
     * @var \Reliese\Meta\SchemaManager
     */
    protected $schemas;

    /**
     * @var \Illuminate\Filesystem\Filesystem
     */
    protected $files;

    /**
     * @var \Reliese\Support\Classify
     */
    protected $class;

    /**
     * @var \Reliese\Coders\Model\Config
     */
    protected $config;

    /**
     * @var \Reliese\Coders\Model\ModelManager
     */
    protected $models;

    /**
     * @var \Reliese\Coders\Model\Mutator[]
     */
    protected $mutators = [];

    /**
     * ModelsFactory constructor.
     *
     * @param \Illuminate\Database\DatabaseManager $db
     * @param \Illuminate\Filesystem\Filesystem $files
     * @param \Reliese\Support\Classify $writer
     * @param \Reliese\Coders\Model\Config $config
     */
    public function __construct(DatabaseManager $db, Filesystem $files, Classify $writer, Config $config)
    {
        $this->db = $db;
        $this->files = $files;
        $this->config = $config;
        $this->class = $writer;
    }

    /**
     * @return \Reliese\Coders\Model\Mutator
     */
    public function mutate()
    {
        return $this->mutators[] = new Mutator();
    }

    /**
     * @return \Reliese\Coders\Model\ModelManager
     */
    protected function models()
    {
        if (! isset($this->models)) {
            $this->models = new ModelManager($this);
        }

        return $this->models;
    }

    /**
     * Select connection to work with.
     *
     * @param string $connection
     *
     * @return $this
     */
    public function on($connection = null)
    {
        $this->schemas = new SchemaManager($this->db->connection($connection));

        return $this;
    }

    /**
     * @param string $schema
     * @param \Reliese\Coders\Console\CodeModelsCommand $command
     */
    public function map($schema, $command)
    {
        if (! isset($this->schemas)) {
            $this->on();
        }

        $mapper = $this->makeSchema($schema);

        foreach ($mapper->tables() as $blueprint) {
            if ($this->shouldNotExclude($blueprint)) {
                $this->create($mapper->schema(), $blueprint->table(), $command);
            }
        }
    }

    /**
     * @param \Reliese\Meta\Blueprint $blueprint
     *
     * @return bool
     */
    protected function shouldNotExclude(Blueprint $blueprint)
    {
        foreach ($this->config($blueprint, 'except', []) as $pattern) {
            if (Str::is($pattern, $blueprint->table())) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param string $schema
     * @param string $table
     * @param \Reliese\Coders\Console\CodeModelsCommand $command
     * @param string|null $dir_prefix
     */
    public function create($schema, $table, $command, $dir_prefix = null)
    {


        $model = $this->makeModel($schema, $table);
        $template = $this->prepareTemplate($model, 'model');

        $file = $this->fillTemplate($template, $model, $dir_prefix);

        if(strlen($dir_prefix) > 0){
            $base_dir = $this->path([$dir_prefix, "Base"]);
        }else{
            $base_dir = "Base";
        }
        // jddd($base_dir);

        $dest_file = $this->modelPath($model, $model->usesBaseFiles() ? [$base_dir] : []);

        if (! $command->confirmToProceed($dest_file)) {
            return false;
        }

        $this->files->put($dest_file, $file);

        if ($this->needsUserFile($model)) {
            $this->createUserFile($model, $dir_prefix);
        }
    }

    /**
     * @param string $schema
     * @param string $table
     *
     * @param bool $withRelations
     *
     * @return \Reliese\Coders\Model\Model
     */
    public function makeModel($schema, $table, $withRelations = true)
    {
        return $this->models()->make($schema, $table, $this->mutators, $withRelations);
    }

    /**
     * @param string $schema
     *
     * @return \Reliese\Meta\Schema
     */
    public function makeSchema($schema)
    {
        return $this->schemas->make($schema);
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     * @todo: Delegate workload to SchemaManager and ModelManager
     *
     * @return array
     */
    public function referencing(Model $model)
    {
        $references = [];

        // TODO: SchemaManager should do this
        foreach ($this->schemas as $schema) {
            $references = array_merge($references, $schema->referencing($model->getBlueprint()));
        }

        // TODO: ModelManager should do this
        foreach ($references as &$related) {
            $blueprint = $related['blueprint'];
            $related['model'] = $model->getBlueprint()->is($blueprint->schema(), $blueprint->table())
                                ? $model
                                : $this->makeModel($blueprint->schema(), $blueprint->table(), false);
        }

        return $references;
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     * @param string $name
     *
     * @return string
     */
    protected function prepareTemplate(Model $model, $name)
    {
        $defaultFile = $this->path([__DIR__, 'Templates', $name]);
        $file = $this->config($model->getBlueprint(), "*.template.$name", $defaultFile);

        return $this->files->get($file);
    }

    /**
     * @param string $template
     * @param \Reliese\Coders\Model\Model $model
     *
     * @return mixed
     */
    protected function fillTemplate($template, Model $model, $dir_prefix = null)
    {
        $template = str_replace('{{date}}', Carbon::now()->toRssString(), $template);
        $template = str_replace('{{modelVersion}}', \Reliese\Coders\CodersServiceProvider::$txdGeneratorVersion, $template);
        $template = str_replace('{{namespace}}', $model->getBaseNamespace($dir_prefix), $template);
        $template = str_replace('{{parent}}', $model->getParentClass(), $template);
        $template = str_replace('{{properties}}', $this->properties($model), $template);
        $template = str_replace('{{class}}', $model->getClassName(), $template);
        $template = str_replace('{{body}}', $this->body($model), $template);

        return $template;
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     *
     * @return string
     */
    protected function properties(Model $model)
    {
        // Process property annotations
        $annotations = '';

        foreach ($model->getProperties() as $name => $hint) {
            $annotations .= $this->class->annotation('property', "$hint \$$name");
        }

        if ($model->hasRelations()) {
            // Add separation between model properties and model relations
            $annotations .= "\n * ";
        }

        foreach ($model->getRelations() as $name => $relation) {
            // TODO: Handle collisions, perhaps rename the relation.
            if ($model->hasProperty($name)) {
                continue;
            }
            $annotations .= $this->class->annotation('property', $relation->hint()." \$$name");
        }

        return $annotations;
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     *
     * @return string
     */
    protected function body(Model $model)
    {
        $body = '';

        foreach ($model->getTraits() as $trait) {
            $body .= $this->class->mixin($trait);
        }

        if ($model->hasCustomCreatedAtField()) {
            $body .= $this->class->constant('CREATED_AT', $model->getCreatedAtField());
        }

        if ($model->hasCustomUpdatedAtField()) {
            $body .= $this->class->constant('UPDATED_AT', $model->getUpdatedAtField());
        }

        if ($model->hasCustomDeletedAtField()) {
            $body .= $this->class->constant('DELETED_AT', $model->getDeletedAtField());
        }

        $body = trim($body, "\n");
        // Separate constants from fields only if there are constants.
        if (! empty($body)) {
            $body .= "\n";
        }

        // Append connection name when required
        if ($model->shouldShowConnection()) {
            $body .= $this->class->field('connection', $model->getConnectionName());
        }

        // When table is not plural, append the table name
        if ($model->needsTableName()) {
            $body .= $this->class->field('table', $model->getTableForQuery());
        }

        if ($model->hasCustomPrimaryKey()) {
            $body .= $this->class->field('primaryKey', $model->getPrimaryKey());
        }

        if ($model->doesNotAutoincrement()) {
            $body .= $this->class->field('incrementing', false, ['visibility' => 'public']);
        }

        if ($model->hasCustomPerPage()) {
            $body .= $this->class->field('perPage', $model->getPerPage());
        }

        if (! $model->usesTimestamps()) {
            $body .= $this->class->field('timestamps', false, ['visibility' => 'public']);
        }

        if ($model->hasCustomDateFormat()) {
            $body .= $this->class->field('dateFormat', $model->getDateFormat());
        }

        if ($model->doesNotUseSnakeAttributes()) {
            $body .= $this->class->field('snakeAttributes', false, ['visibility' => 'public static']);
        }

        if ($model->hasCasts()) {
            $body .= $this->class->field('casts', $model->getCasts(), ['before' => "\n"]);
        }

        if ($model->hasDates()) {
            $body .= $this->class->field('dates', $model->getDates(), ['before' => "\n"]);
        }

        if ($model->hasHidden() && $model->doesNotUseBaseFiles()) {
            $body .= $this->class->field('hidden', $model->getHidden(), ['before' => "\n"]);
        }

        if ($model->hasFillable() && $model->doesNotUseBaseFiles()) {
            $body .= $this->class->field('fillable', $model->getFillable(), ['before' => "\n"]);
        }

        if ($model->hasHints() && $model->usesHints()) {
            $body .= $this->class->field('hints', $model->getHints(), ['before' => "\n"]);
        }
		
		//aggiungiamo il nostro array dei tipi
		$arr_tipi = [];
		
		foreach($model->getDates() as $field){
			$arr_tipi[$field] = "date";
		}
		
		//dato che i campi datetime sono contenuti anche nell'array dates, mettendoci dopo andiamo di fatto a sovrascrivere il comportamento precedente
		foreach($model->getDateTimes() as $field){
			$arr_tipi[$field] = "datetime";
        }
        
		//dato che i campi time sono contenuti anche nell'array dates, mettendoci dopo andiamo di fatto a sovrascrivere il comportamento precedente
		foreach($model->getTimes() as $field){
			$arr_tipi[$field] = "time";
		}
		
		$body .= $this->class->field('types', $arr_tipi, ['before' => "\n\t//array con i tipi per la conversione dei dati\n"]);
        
        //aggiungiamo le regole legate ai limiti del database
        $body .= $this->class->field('db_rules', $model->txdRules, ['before' => "\n\t//Contiene le regole di validazione dei campi legate al DB (eg. numeric|max, etc...)\n\t//l'elenco completo dei controlli aggiunti da noi e' nel jsmServiceProvider\n"]);

        foreach ($model->getMutations() as $mutation) {
            $body .= $this->class->method($mutation->name(), $mutation->body(), ['before' => "\n"]);
        }

        foreach ($model->getRelations() as $constraint) {
			
			$reflect = new \ReflectionClass($constraint);
			$belongs_to = $reflect->getShortName() == "BelongsTo" ? true : false;
			
            $body .= $this->class->method($constraint->name(), $constraint->body(), ['before' => "\n", "belongs_to" => $belongs_to]);
        }

        // Make sure there not undesired line breaks
        $body = trim($body, "\n");

        return $body;
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     *
     * @param array $custom
     *
     * @return string
     */
    protected function modelPath(Model $model, $custom = [])
    {
        $modelsDirectory = $this->path(array_merge([$this->config($model->getBlueprint(), 'path')], $custom));

        if (! $this->files->isDirectory($modelsDirectory)) {
            $this->files->makeDirectory($modelsDirectory, 0755, true);
        }

        return $this->path([$modelsDirectory, $model->getClassName().'.php']);
    }

    /**
     * @param array $pieces
     *
     * @return string
     */
    protected function path($pieces)
    {
        return implode(DIRECTORY_SEPARATOR, (array) $pieces);
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     *
     * @return bool
     */
    public function needsUserFile(Model $model)
    {
        return ! $this->files->exists($this->modelPath($model)) && $model->usesBaseFiles();
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     */
    protected function createUserFile(Model $model, $dir_prefix = null)
    {
        if(strlen($dir_prefix) > 0){
            $file = $this->modelPath($model, [$dir_prefix]);
        }else{
            $file = $this->modelPath($model);
        }
        // jddd($file);

        $template = $this->prepareTemplate($model, 'user_model');
        $template = str_replace('{{namespace}}', $model->getNamespace().(strlen($dir_prefix) > 0 ? "\\".$dir_prefix : ""), $template);
        $template = str_replace('{{class}}', $model->getClassName(), $template);
        $template = str_replace('{{parent}}', '\\'.$model->getBaseNamespace().'\\'.$model->getClassName(), $template);
        $template = str_replace('{{body}}', $this->userFileBody($model), $template);

        $this->files->put($file, $template);
    }

    /**
     * @param \Reliese\Coders\Model\Model $model
     *
     * @return string
     */
    protected function userFileBody(Model $model)
    {
        $body = '';

        $body .= '
	public function __construct(array $attributes = array()){
        parent::__construct($attributes);
        
        /* qui possiamo aggiungere o sovrascrivere i tipi (di seguito alcuni esempi di campi supportati) */

        //$this->types["float_field_name"] = "float";
        //$this->types["valuta_field_name"] = "valuta";
        //$this->types["date_field_name"] = "date";
        //$this->types["datetime_field_name"] = "datetime";
    }
    
';

        if ($model->hasHidden()) {
            $body .= $this->class->field('hidden', $model->getHidden());
        }

        if ($model->hasFillable() && false) { //con la nuova versione degli attributi i fillable devono essere vuoti perche' vengono composti dinamicamente
            $body .= $this->class->field('fillable', $model->getFillable(), ['before' => "\n"]);
        }
		
		/*
		 * aggiungiamo i placeholder per i due array che ci servono per la validazione e la formattazione dei dati
		 */
		// $body .= $this->class->field('types', [], ['before' => "\n\t/* Prima di decommentare, controllare di non perdere info dal modello base! */\n/*", 'after' => "*/\n"]);
		$body .= $this->class->field('rules', [], [
            'before' => "\n\t/* \n\t * DEPRECATO -- usare gli opportuni metodi di txdAttributes\n\t *\n\t//Contiene le regole di validazione dei campi (eg. required|date_nullable|numeric, etc...)\n\t//l'elenco completo dei controlli aggiunti da noi e' nel jsmServiceProvider\n",
            'after' => "*/\n"
        ]);

        $body .= '
    /**
     * popola l\'elenco degli attributi legati al database
     * 
     */
    public function db_fields_definition($db_fields) {

        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        
        /** ESEMPIO */
        // $db_fields->add("nome")->add_settore("anagrafica")->add_rule("required");
        // $db_fields->add("cognome")->add_settore("anagrafica");


    }
    
';

        // Make sure there is not an undesired line break at the end of the class body
        $body = ltrim(rtrim($body, "\n"), "\n");

        return $body;
    }

    /**
     * @param \Reliese\Meta\Blueprint|null $blueprint
     * @param string $key
     * @param mixed $default
     *
     * @return mixed|\Reliese\Coders\Model\Config
     */
    public function config(Blueprint $blueprint = null, $key = null, $default = null)
    {
        if (is_null($blueprint)) {
            return $this->config;
        }

        return $this->config->get($blueprint, $key, $default);
    }
}
