<?php

namespace Txd\Api\Clients\Pipes;

use Illuminate\Support\Facades\Cache;
use Prewk\Result;
use Txd\Api\Clients\Contracts\AuthStrategy;
use Txd\Api\Clients\Contracts\RequestPipe;
use Txd\Api\Clients\RequestParams;

class WithCache implements RequestPipe
{
    public $order = 10;

    public function order(): int
    {
        return $this->order;
    }

    public function __construct(protected int $ttl, protected ?string $cacheKey, protected ?AuthStrategy $auth, protected string $clientClass) {}

    public function handle(RequestParams $request, $next): Result
    {
        $key = $this->cacheKey ?? $this->buildCacheKey($request->method, $request->path, $request->options);

        if (Cache::has($key)) {

            return Ok(Cache::get($key));
        }

        $response = $next($request);

        if ($response->isOk()) {
            Cache::put(
                $key,
                $response->unwrap(),
                $this->ttl
            );

        }

        return $response;

    }

    protected function buildCacheKey(
        string $method,
        string $path,
        mixed $payload
    ): string {

        $identity = $this->auth?->cacheDiscriminator() ?? 'public';
        $hash = is_string($payload)
            ? $payload
            : hash('sha256', json_encode($payload, JSON_THROW_ON_ERROR));

        return sprintf(
            'txd-api-client.%s.%s.%s.%s.%s',
            $this->clientClass,
            $method,
            $identity,
            md5($path),
            $hash
        );
    }
}
