<?php

namespace Txd\Api\Clients;

use Closure;
use Exception;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Support\Facades\Http;
use Prewk\Result;
use Txd\Api\Clients\Contracts\AuthStrategy;
use Txd\Api\Clients\Contracts\RequestPipe;
use Txd\Api\Clients\Pipes\AsJson;
use Txd\Api\Clients\Pipes\FromPath;
use Txd\Api\Clients\Pipes\Throws;
use Txd\Api\Clients\Pipes\WithCache;

class RetryException extends Exception {}

class HttpClient
{
    protected ?AuthStrategy $auth = null;

    protected array $pipeline = [];

    public function __construct(protected array $config = []) {
        
    }

    public function withAuth(AuthStrategy $auth): static
    {
        $this->auth = $auth;

        return $this;
    }

    public function cached(?int $ttl = null, ?string $customCacheKey = null,bool $renew=false): static
    {
        $this->addPipe(new WithCache($ttl ?? 300, $customCacheKey, $this->auth, static::class, renew: $renew));

        return $this;
    }

    public function throw(): static
    {
        $this->addPipe(new Throws);

        return $this;
    }

    public function asJson($associative = true): static
    {
        $this->addPipe(new AsJson($associative));

        return $this;
    }

    public function addPipe(RequestPipe $pipe)
    {
        $this->pipeline[] = $pipe;
    }
    
    protected ?Closure $pipelineTransform = null;
    
    /**
     * 
     * @param Closure(array<int,RequestPipe>):array<int,RequestPipe> $callback 
     * @return $this 
     */
    public function transformPipeline(Closure $callback)
    {
        $this->pipelineTransform = $callback;
        return $this;
    }

    protected function prepare(): PendingRequest
    {
        $req = Http::timeout(300)
            ->baseUrl($this->config['host']);
            
        if(array_key_exists("base_path",$this->config)){
            $this->addPipe(new FromPath($this->config["base_path"]));
        }

        if ($this->auth) {
            $req = $this->auth->authenticate($req);
        }

        return $req;
    }

    protected function buildPipeline()
    {
        if(!is_null($this->pipelineTransform)){
            $this->pipeline = ($this->pipelineTransform)($this->pipeline);
        }
        usort($this->pipeline, fn ($a, $b) => $a->order() <=> $b->order());

        return new Pipeline()
            ->through($this->pipeline);
    }

    public function send(string $method, string $path, array $options = []): ResponseResult
    {
        try {
            return $this->attempt($method, $path, $options,true);
        } catch (RetryException $e) {
            return $this->attempt($method, $path, $options);
        } finally {
            $this->pipeline = [];
        }
    }

    protected function attempt(string $method, string $path, array $options = [],bool $retry = false): ResponseResult
    {

        $request = new RequestParams($this->prepare(), $method, $path, $options);

        $pipe = $this->buildPipeline();
        
        return $pipe->send($request)
            ->then(function ($pipedRequest) use ($retry) {
                $response = $pipedRequest->request->send($pipedRequest->method, $pipedRequest->path, $pipedRequest->options);

                if ($response->unauthorized() && $retry && $this->auth?->handleUnauthorized($this)) {
                    throw new RetryException;
                }

                return ResponseResult::fromResponse($response->ok()
                    ? Ok($response->body())
                    : Err($response->body()),$response);
            });

    }

    public function get(string $path, array $query = []): ResponseResult
    {
        return $this->send('GET', $path, ['query' => $query]);
    }

    public function post(string $path, array $data = []): ResponseResult
    {
        return $this->send('POST', $path, ['json' => $data]);
    }

    public function postForm(string $path, array $data = []): ResponseResult
    {
        return $this->send('POST', $path, ['form_params' => $data]);
    }
}
