<?php

namespace Txd\Api\Clients\Pipes;

use Illuminate\Support\Facades\Cache;
use Prewk\Result;
use Txd\Api\Clients\Contracts\AuthStrategy;
use Txd\Api\Clients\Contracts\RequestPipe;
use Txd\Api\Clients\RequestParams;
use Txd\Api\Clients\ResponseResult;

class WithCache implements RequestPipe
{
    public $order = 50;

    public function order(): int
    {
        return $this->order;
    }

    public function __construct(protected int $ttl, protected ?string $cacheKey, protected ?AuthStrategy $auth, protected string $clientClass, protected bool $renew) {}

    public function handle(RequestParams $request, $next): ResponseResult
    {
        $key = $this->cacheKey ?? $this->buildCacheKey($request->method, $request->path, $request->options);

        if (!$this->renew && Cache::has($key)) {
            $cachedResponse = Cache::get($key);
            return new ResponseResult(
                Ok($cachedResponse["value"]),
                null,
                $cachedResponse["headers"],
                $cachedResponse["status"]
            );
        }

        $response = $next($request);

        if ($response->isOk()) {
            Cache::put(
                $key,
                [
                    "value" => $response->unwrap(),
                    "headers" => $response->headers(),
                    "status" => $response->status(),
                ],
                $this->ttl
            );

        }

        return $response;

    }

    protected function buildCacheKey(
        string $method,
        string $path,
        mixed $payload
    ): string {

        $identity = $this->auth?->cacheDiscriminator() ?? 'public';
        $hash = is_string($payload)
            ? $payload
            : hash('sha256', json_encode($payload, JSON_THROW_ON_ERROR));

        return sprintf(
            'txd-api-client.%s.%s.%s.%s.%s',
            $this->clientClass,
            $method,
            $identity,
            md5($path),
            $hash
        );
    }
}
