<?php

namespace Txd\Api\Clients\Auth;

use Exception;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Support\Facades\Http;
use Txd\Api\Clients\Contracts\AuthStrategy;
use Txd\Api\Clients\HttpClient;

use function Illuminate\Filesystem\join_paths;

abstract class AbstractOauthStrategy implements AuthStrategy
{
    public function __construct(
        protected array $config
    ) {}

    abstract protected function getAccessToken(): TokenStore;

    abstract protected function storeAccessToken(TokenStore $store): void;

    public function authenticate(PendingRequest $request): PendingRequest
    {
        return $request->withToken(
            $this->getAccessToken()->access_token
        );
    }

    /**
     * @return bool true if credentials were renewed and retry should occur
     *
     * @throws Exception when refresh attempt fails
     */
    public function handleUnauthorized(HttpClient $client): bool
    {

        $token = $this->refreshAccessToken();
        if (is_null($token)) {
            return false;
        }

        $this->storeAccessToken($token);

        return true;
    }

    abstract protected function refreshTokenPayload(): ?array;

    protected function refreshAccessToken(): ?TokenStore
    {
        $refreshPayload = $this->refreshTokenPayload();

        if (is_null($refreshPayload)) {
            return null;
        }

        $payload = [
            'client_id' => $this->config['client_id'],
            'client_secret' => $this->config['client_secret'],
            ...$refreshPayload,
        ];

        $response = Http::asForm()
            ->timeout(30)
            ->post(join_paths($this->config['host'], $this->config['token_path'] ?? 'oauth/token'), $payload);

        if (!$response->ok()) {
            throw new Exception(
                'OAuth token refresh failed: '.
                $response->body()
            );

            return null;
        }

        $data = $response->json();

        return TokenStore::fromArray($data);

    }
}
