<?php

namespace txd\ExportJobsManager;

class txdExportManager{

    public $job_type;
    public $disk_name;
    public $job_prefix;
    public $out_dir;

    protected $file_list = [];
    protected $disk;
    protected $config_info = [];
    protected $s3_obj;

    protected $data_raw;
    protected $formatted_data_raw;

    protected $documentazione_attiva = false;

    public function __construct($type, $out_dir=null)
    {

        /*
		 * log documentazione
		 */
		if(class_exists("jsmDoc")){

			try{
				\jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__);
				$this->documentazione_attiva = true;
			}catch(\Exception $ex){
				\Log::error(__FILE__." | ".__FUNCTION__." | Errore creazione documentazione: ".$ex->getMessage());
			}
		}

        $this->job_type = $type;
        $this->disk_name = config("txd_export_jobs_manager.export_disk_name");
        $this->disk = \Storage::disk($this->disk_name);
        $this->s3_obj = new \AWS_S3($this->disk_name);

        if(is_null($out_dir)){
            $out_dir = "out-".str_slug(config("app.name"));
        }

        $this->out_dir = $out_dir;

        switch ($type) {
            case 'Excel':
                $this->init_excel($this->out_dir);
                break;

            case 'PDF':
                $this->init_pdf($this->out_dir);
                break;
            
            default:
                throw new \Exception("Export type not supported: ".$type, 1);
        }

    }


    protected function init_excel($out_dir){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->config_info = [
            "job_type" => "Excel",
            "source" => [],
            "prefix" => "",
            "out_dir" => $out_dir,
        ];
    }

    protected function init_pdf($out_dir){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->config_info = [
            "job_type" => "PDF",
            "source" => [],
            "prefix" => "",
            "out_dir" => $out_dir,
            "options" => [
                "page-size" => "A4",
                "margin-top" => "0.4in",
                "margin-right" => "0.4in",
                "margin-bottom" => "0.4in",
                "margin-left" => "0.4in",
                "encoding" => "UTF-8",
                "no-outline" => null,
                // "footer-html" => null,
            ]
        ];
    }
    

    public function add_source_file($content){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $filename = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_").".html";

        $this->disk->put($filename, $content);
        $this->file_list[] = $filename;
    }


    public function add_pdf_footer($content){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->job_type !== "PDF"){
            throw new \Exception(__FUNCTION__." | Method available only for PDF Exports");
        }

        $filename = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_")."-footer.html";

        $this->disk->put($filename, $content);
        
        $this->config_info["options"]["footer-html"] = $this->s3_obj->getPresignedURL($filename, null, "+600 minutes");
    }


    public function start(){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->job_prefix = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_");

        $this->config_info["prefix"] = $this->job_prefix;

        foreach($this->file_list as $file){
            $this->config_info["source"][] = $this->s3_obj->getPresignedURL($file, null, "+600 minutes");
        }

        $this->disk->put($this->job_prefix.".json", json_encode($this->config_info));
    }


    public function job_terminated($key = null){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        switch ($this->job_type) {
            case 'Excel':
                $estensione = ".pdf";
                break;

            case 'PDF':
                $estensione = ".pdf";
                break;
            
            default:
                $estensione = ".null";
                break;
        }

        if(strlen($key) == 0){
            $key = $this->job_prefix;
        }

        return $this->s3_obj->isFile($key.$estensione, "../".$this->out_dir);
    }

    


    public function get_exported_file_url($key = null, $expiry = "+10 minutes"){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        switch ($this->job_type) {
            case 'Excel':
                $estensione = ".pdf";
                break;

            case 'PDF':
                $estensione = ".pdf";
                break;
            
            default:
                $estensione = ".null";
                break;
        }

        if(strlen($key) == 0){
            $key = $this->job_prefix;
        }

        return $this->s3_obj->getPresignedURL($key.$estensione, "../".$this->out_dir, $expiry);
    }

}