<?php

namespace txd\ExportJobsManager;

use \txd\ExportJobsManager\ModuleConfig\pdfConfig;
class txdExportManager{

    public static $show_in_browser = true;

    public $job_type;
    public $disk_name;
    public $job_prefix;
    public $out_dir;
    public $estensione_file = ".null";

    protected $file_list = [];
    protected $disk;
    protected $config_info = [];
    protected $s3_obj;

    protected $data_raw;
    protected $formatted_data_raw;
    protected $link_expiry = "+10 minutes";

    protected $documentazione_attiva = false;

    /**
     *
     * @var pdfConfig
     */
    protected $pdf_config;

    public function __construct($type, $out_dir=null)
    {

        /*
		 * log documentazione
		 */
		if(class_exists("jsmDoc")){

			try{
				\jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__);
				$this->documentazione_attiva = true;
			}catch(\Exception $ex){
				\Log::error(__FILE__." | ".__FUNCTION__." | Errore creazione documentazione: ".$ex->getMessage());
			}
        }
        
        if(config("app.env") == "home" || config("app.env") == "local"){
            $this->link_expiry = "+600 minutes";
        }

        $this->job_type = $type;
        $this->disk_name = config("txd_export_jobs_manager.export_disk_name");
        $this->disk = \Storage::disk($this->disk_name);
        $this->s3_obj = new \AWS_S3($this->disk_name);

        if(is_null($out_dir)){
            $out_dir = "out-".str_slug(config("app.name"));
        }

        $this->out_dir = $out_dir;

        switch ($type) {
            case 'Excel':
                $this->init_excel($this->out_dir);
                $this->estensione_file = ".xlsx";
                break;

            case 'PDF':
                $this->init_pdf($this->out_dir);
                $this->estensione_file = ".pdf";
                break;
            
            default:
                throw new \Exception("Export type not supported: ".$type, 1);
        }

    }


    protected function init_excel($out_dir){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->config_info = [
            "job_type" => "Excel",
            "source" => [],
            "prefix" => "",
            "out_dir" => $out_dir,
        ];
    }

    protected function init_pdf($out_dir){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->pdf_config = new pdfConfig();

        $this->config_info = [
            "job_type" => "PDF",
            "source" => [],
            "prefix" => "",
            "out_dir" => $out_dir,
            "options" => []
        ];
    }
    

    public function add_source_content($content){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $filename = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_").".html";

        $this->disk->put($filename, $content);
        $this->file_list[] = $filename;
    }


    public function add_pdf_footer($content){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->job_type !== "PDF"){
            throw new \Exception(__FUNCTION__." | Method available only for PDF Exports");
        }

        $filename = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_")."-footer.html";

        $this->disk->put($filename, $content);
        
        $this->config_info["options"]["footer-html"] = $this->s3_obj->getPresignedURL($filename, null, $this->link_expiry);
    }

    public function add_pdf_header($content){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->job_type !== "PDF"){
            throw new \Exception(__FUNCTION__." | Method available only for PDF Exports");
        }

        $filename = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_")."-header.html";

        $this->disk->put($filename, $content);
        
        $this->config_info["options"]["header-html"] = $this->s3_obj->getPresignedURL($filename, null, $this->link_expiry);
    }

    /**
     *
     * @return pdfConfig
     */
    public function get_pdf_config(){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->job_type !== "PDF"){
            throw new \Exception(__FUNCTION__." | Method available only for PDF Exports");
        }

        return $this->pdf_config;
    }

    public function set_pdf_config(pdfConfig $config){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
        
        $this->pdf_config = $config;
    }


    public function start(){

        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->job_prefix = str_slug(date("YmdHis-").microtime()."-".hash("sha1", config("app.name")."-".\Uuid::generate(4)->string), "_");

        $this->config_info["prefix"] = $this->job_prefix;

        foreach($this->file_list as $file){
            $this->config_info["source"][] = $this->s3_obj->getPresignedURL($file, null, $this->link_expiry);
        }
        
        $this->config_info["options"] = array_merge($this->config_info["options"], $this->pdf_config->toArray());
        $this->disk->put($this->job_prefix.".json", json_encode($this->config_info));
    }


    public function job_terminated($key = null){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if(strlen($key) == 0){
            $key = $this->job_prefix;
        }

        return $this->s3_obj->isFile($key.$this->estensione_file, "../".$this->out_dir);
    }

    


    public function get_exported_file_url($key = null, $expiry = "+10 minutes"){
        if($this->documentazione_attiva){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if(strlen($key) == 0){
            $key = $this->job_prefix;
        }

        return $this->s3_obj->getPresignedURL($key.$this->estensione_file, "../".$this->out_dir, $expiry);
    }

}