<?php

namespace Txd;

use Carbon\Carbon;
use Exception;
use Txd\FieldTypes\Hidden;
use Txd\Traits\FieldTypes\AddTypesTrait;


/**
 * Helper per definizione, costruzione e validazione di form complessi
 */
class txdFields implements \IteratorAggregate,\ArrayAccess {
    public static $AUTO_APPLY_DB_FIELDS = true;    

    private static $legacyTypesConversion = [
        "text" => FieldTypes\Text::class,
        "data" => FieldTypes\Date::class,
        "datatime" => FieldTypes\Datetime::class,
        "valuta" => FieldTypes\Valuta::class,
        "select" => FieldTypes\Select::class,
        "select_linked" => FieldTypes\SelectLinked::class,
        "txdJson" => FieldTypes\TxdJson::class,
        "email" => FieldTypes\Email::class,
        "password" => FieldTypes\Password::class,
        "number" => FieldTypes\Number::class,
        "checkbox" => FieldTypes\LegacyCheckbox::class,
        "textarea" => FieldTypes\Textarea::class,
        "html" => FieldTypes\Html::class,
        "radio" => FieldTypes\Radio::class,
        "hidden" => FieldTypes\Hidden::class,
        "parse_value" => FieldTypes\ParseValue::class,
        "custom_macro" => FieldTypes\CustomMacro::class,
        "file" => FieldTypes\File::class,
        "solo_testo" => FieldTypes\SoloTesto::class,

    ];

    use AddTypesTrait;
	
    public $throwaway;
    public $source;

    protected $sectors = [
        "default" => []
    ];
    protected $skip_auto_merge_sectors = [];
    
    protected $current_model_obj;


    public function __construct($current_model_obj = null,$throwaway = false){
		$this->current_model_obj = $current_model_obj;
        $this->throwaway = $throwaway;
	}
    
    // implementazione arrayAccess e IteratorAggregate
    // funzioni per usare un oggetto txdFields come un array. tutti questi metodi funzionano solo sul settore "default"
    public function offsetSet($offset, $value) {
        if (is_null($offset)) {
            $this->sectors["default"][] = $value;
        } else {
            $this->sectors["default"][$offset] = $value;
        }
    }

    public function offsetExists($offset) {
        return isset($this->sectors["default"][$offset]);
    }

    public function offsetUnset($offset) {
        unset($this->sectors["default"][$offset]);
    }

    public function offsetGet($offset) {
        return isset($this->sectors["default"][$offset]) ? $this->sectors["default"][$offset] : null;
    }

    
    public function getIterator() {
		return new \ArrayIterator($this->sectors["default"]);
    }
    // fine implementazione ArrayAccess e IteratorAggregate

    

    /**
     * inizializzazione statica di un oggetto txdFields passando direttamente l'array del settore "default"
     * 
     * @param mixed $current_model_obj si deve comportare come un oggetto (attributi accessibili tramite ->) oppure essere null
     * @param array $new_item_list 
     * 
     * @return \Txd\txdFields
     */
    protected static function init($current_model_obj,$new_item_list){
        $fields = new txdFields($current_model_obj);
        $fields->sectors["default"] = $new_item_list;
        return $fields;
    }

    /**
     * ritorna un nuovo txdFields con il settore default popolato con il merge dei settori selezionati del txdfields corrente
     * 
     * @param array|string $settori
     * 
     * @return \Txd\txdFields
     */
    public function reduce($settori = []){
        if(is_string($settori)){
            $settori = [$settori];
        }
        $new_item_list = [];
        
        if(count($settori) == 0){
            $sectors_to_skip = array_keys(
                                array_filter(
                                    $this->skip_auto_merge_sectors,
                                    function($el){
                                        return !$el;
                                    }
                                )
                            );

            $new_item_list = array_merge(
                ...array_values(
                    array_filter(
                        $this->sectors,
                        function($key) use($sectors_to_skip){
                            return !in_array($key,$sectors_to_skip);
                        },
                        ARRAY_FILTER_USE_KEY)
                    )
                );
        }else{
            foreach($settori as $sector){
                if(array_key_exists($sector,$this->sectors)){
                    $new_item_list = array_merge($new_item_list,$this->sectors[$sector]);
                }
            }
        }
        
        return static::init($this->current_model_obj,$new_item_list);
    }

    /**
     * ritorna l'array dei settori con le loro proprietà
     * 
     * 
     * @return Array
     */
    public function getSectors(){
        $out = [];
        foreach(array_keys($this->sectors) as $key){
            $out[$key] = ["automerge"=>!($this->skip_auto_merge_sectors[$key] ?? false)];
        }
        return $out;
    }


    static protected function uasort(array &$array, $value_compare_func)
	{
		$index = 0;
		foreach ($array as &$item) {
			$item = array($index++, $item);
		}

		$result = uasort($array, function($a, $b) use($value_compare_func) {
			$result = call_user_func($value_compare_func, $a[1], $b[1]);
			return $result == 0 ? $a[0] - $b[0] : $result;
		});

		foreach ($array as &$item) {
			$item = $item[1];
		}

		return $result;
	}

    /**
     * ordina i campi del settore selezionato secondo la funzione di sort passata. di default ordina per il campo ->order
     * 
     * @param callable|null $sorting_function
     * @param string|null $sector
     * 
     * @return $this
     */
    public function sort(callable $sorting_function = null,string $sector = "default"){
        if(is_null($sorting_function)){
            $sorting_function = function($a,$b){
                return $a->order - $b->order;
                
                
            };
        }
        
        static::uasort($this->sectors[$sector],$sorting_function);
        return $this;
    }

    /**
     * ritorna l'array del settore selezionato. Ritorna l'array dei settori in caso venga passato Null
     * 
     * @param string|null $sector
     * 
     * @return array
     */
    public function toArray(string $sector = null) {
        if(is_null($sector)){
		    return $this->sectors;
        }
        return $this->sectors[$sector] ?? [];
    }


    public static function findClass(string $typeName,array $additionalNamespaces = []): string {
        
        if(class_exists($typeName)) return $typeName;
        $namespaces = array_merge(["Txd\\FieldTypes",config("txd_fieldtypes.local_fieldtypes_namespace","")],$additionalNamespaces);
        foreach($namespaces as $namespace){
            $class = $namespace."\\".$typeName;
            if(class_exists($class)) return $class;
        }
        // test con testo studly
        $originalTypeName = $typeName;
        $typeName = \Illuminate\Support\Str::studly($typeName);
        foreach($namespaces as $namespace){
            $class = $namespace."\\".$typeName;
            if(class_exists($class)) return $class;
        }
        $typeName = $originalTypeName;

        if(isset(txdFields::$legacyTypesConversion[$typeName])){
            $class = txdFields::$legacyTypesConversion[$typeName];
        }else{
            if(config("app.debug",false)){
                throw new Exception(__METHOD__." - FieldType '".$typeName."' is not a known php class nor a legacy type");
            }else{
                \Log::warning(__METHOD__." - FieldType '".$typeName."' is not a known php class nor a legacy type. '".Text::class."' used instead");
                $class = \Txd\FieldTypes\Text::class;;
            }
        }

        return $class;
    }

    /**
     * aggiunge un attributo all'elenco
     * 
     * @param string $nome nome del campo su DB
     * @param string $tipo tipo di attributo. accetta il nome di una classe in Txd\FieldTypes, una classe custom o uno dei tipi legacy (text, email, password, data, select, etc...)
     * @param string $etichetta [default: ""] eventuale etichetta da stampare
     * @param array $classi_html elenco di classi da passare al campo
     * 
     * @return txdAttributes
     */
    
    public function add($nome,$typeName = "Txd\FieldTypes\Text", $etichetta = "", $classi_html = []){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
        
        $class = static::findClass($typeName);
        
        $tmp = new $class($nome, $etichetta, $classi_html);
        return $this->addField($tmp);
    }

    /**
     * Rimuove il txdAttribute con nome $fieldName dal settore selezionato
     * @param string $fieldName
     * 
     * @return void
     */
    public function remove(string $fieldName,string $sector = "default"){
        if(isset($this->sectors[$sector][$fieldName])) unset($this->sectors[$sector][$fieldName]);
    }
    

    /**
     * aggiunge il txdAttributes passato in ingresso ai settori selezionati (a default se non vengono selezionati settori)
     * @param \Txd\txdAttributes $field
     * @param array|string $sectors
     * 
     * @return $field
     */
    public function addField(\Txd\txdAttributes $field, $sectors = []){
        if(is_string($sectors)){
            $sectors = [$sectors];
        }
        $field->set_current_model_obj($this->current_model_obj);
        $field->set_current_db_fields($this);
        
        if(count($sectors)==0 ){
            $this->sectors["default"][$field->nome_campo] = $field;
        }else{
            foreach($sectors as $sector){
                if(!isset($this->sectors[$sector])){
                    $this->sectors[$sector] = [];
                }
                $this->sectors[$sector][$field->nome_campo] = $field;
            }
        }

        $field->parse_model_rules();
        return $field;
    }

    
    /**
     * ritorna il txdAttribute con nome del campo corrispondente, cercandolo nel settore segnalato (default se null)
     * @param string $fieldName
     * @param string $sector
     * 
     * @return \txd\txdAttributes
     */
    public function retrieve(string $fieldName,string $sector = null){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
        if(is_null($sector)) $sector  = "default";
        return $this->sectors[$sector][$fieldName] ?? null;
    }

    /**
     * imposta il modello di partenza all'interno del campo corrente
     * 
     * @param object|null $obj
     */
    public function set_current_model_obj($obj){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
        $this->current_model_obj = $obj;
        foreach($this->sectors as $chiave => $sector){
            foreach($sector as $key => $field){
                $field->set_current_model_obj($this->current_model_obj);
                $field->set_current_db_fields($this);
            }
        }
    }

    /**
     * invoca il metodo inizializzaCampi del modello su cui sono stati costruiti gli attributi
     * in modo che siano disponibili nell'array db_fields del modello
     * 
     */
    public function build(){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        foreach($this->sectors as $chiave => $sector){
            foreach($sector as $key => $field){
                $field->finalizza_info();
            }
        }
        
        $this->current_model_obj->set_db_fields($this);
        if(self::$AUTO_APPLY_DB_FIELDS){
            $this->current_model_obj->inizializzaCampi();
        }
    }

    /**
     * crea il campo di verifica per ricostruzione univoca del txdFields per la validazione
     * 
     * @return string
     */
    public function verification_field(){
        $random = localUniqueString(null,20);
        $fakeObj = json_decode(json_encode(["txd_verification"=>$random]));
        session()->flash("_old_input.txd_verification",$random);
        $hidden = new Hidden("txd_verification");
        $hidden->set_current_model_obj($fakeObj);
        $sessionStore = static::getVerificationTokens();
        $sessionStore[$random] = ["source"=>$this->source,"exp"=>Carbon::now()->addMinutes(config("txd_fieldtypes.txd_verification_expiration_interval",15))];
        static::setVerificationTokens($sessionStore);
        return $hidden->render_edit();
    }

    /**
     * recupera l'elenco di token di verifica validi dalla sessione
     * @return array
     */
    public static function getVerificationTokens(){
        $now = Carbon::now();
        return array_filter((array)session()->get("txdFieldsVerification"),function($el)use($now){
            return $el["exp"] > $now;
        });
    }

    /**
     * cerca nell'array di token di verifica il token passato. Se lo trova, il token viene invalidato e ritorna l'array di settori corrispondente
     * @param string $token
     * 
     * @return array|null
     */
    public static function getVerificationTokenMatch(string $token){
        $sessionStore = txdFields::getVerificationTokens();
        $tokenMatch = array_pull($sessionStore,$token);
        txdFields::setVerificationTokens($sessionStore);
        if(!is_null($tokenMatch)){
            return $tokenMatch["source"];
        }
        return null;
    }

    /**
     * salva in sessione l'array di token di verifica validi
     * @param array $tokensArray
     * 
     * @return void
     */
    public static function setVerificationTokens($tokensArray){
        session()->put("txdFieldsVerification", $tokensArray);
    }


    

    /**
     * ritorna l'istanza corrente (utile per Facade)
     * 
     * @return $this
     */
    public function getInstance(){
        return $this;
    }

    /**
     * Aggiunge un settore identificato da $sector. Il settore viene costruito eseguendo la funzione $sector_definition.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti. 
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     * @param string $sector
     * @param \Closure $sector_definition
     * @param bool $add_to_parent_sector
     * 
     * @return $this
     */
    public function addSector(string $sector, \Closure $sector_definition,bool $add_to_parent_sector = false){
        $current = resolve("dbFields");
        $new = new txdFields($this->current_model_obj);
        app()->instance("dbFields",$new);
        
        
        $sector_definition($new);
        $new->sort();
        if(!isset($this->sectors[$sector])){
            $this->sectors[$sector] = [];
        }
        foreach($new->toArray() as $sectorName=>$sectorFields){
            // if($sectorName === "default"){
            //     $sectorName =$sector;
            //     if($add_to_parent_sector){
            //         foreach($sectorFields as $key=>$field){
            //             $this->addField($field,"default");
            //         }        
            //     }
            // }
            $sectors = [$sectorName];
            if($sectorName === "default"){
                $sectors = [$sector];
                if($add_to_parent_sector){
                    $sectors =["default",$sector];
                }
            }
            foreach($sectorFields as $key=>$field){
                $this->addField($field,$sectors);
            }
        };
        
        if($current->throwaway){
            \Txd\Provider\FieldTypesServiceProvider::registerDbFields();
        }else{
            app()->instance("dbFields",$current);
        }
        return $this;
    }

    /**
     * elimina il settore indicato
     * 
     * @param string $sector
     * 
     * @return $this
     */
    public function deleteSector(string $sector){
        if(isset($this->sectors[$sector])){
            unset($this->sectors[$sector]);
        }
        return $this;
    }

    /**
     * modifica il settore identificato da $sector. Il settore viene passato alla funzione $sector_definition per essere modificato.
     * il contenuto precedente del settore viene completamente sostituito dalle nuove definizioni.
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     * @param string $sector
     * @param \Closure $sector_definition
     * 
     * @return $this
     */
    public function editSector(string $sector, \Closure $sector_definition){
        $current = resolve("dbFields");
        $new = static::init($this->current_model_obj,$this->toArray($sector));
        app()->instance("dbFields",$new);
        
        $sector_definition($new);
        $new->sort();
        
        if(isset($this->sectors[$sector])){
            unset($this->sectors[$sector]);
        }

        $this->sectors[$sector]= [];
        foreach($new->toArray() as $sectorName=>$sectorFields){
            if($sectorName === "default") $sectorName =$sector;
            foreach($sectorFields as $key=>$field){
                $this->addField($field,$sectorName);
            }
        };
        
        if($current->throwaway){
            \Txd\Provider\FieldTypesServiceProvider::registerDbFields();
        }else{
            app()->instance("dbFields",$current);
        }
        return $this;
    }

    /**
     * Copia il contenuto del settore $source in un settore identificato da $sector. Il settore viene costruito eseguendo la funzione $sector_definition.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti.
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     * 
     * @param string $sector
     * @param string $source
     * @param \Closure|null $sector_definition
     * 
     * @return $this
     */
    public function cloneSector(string $source,string $new_sector, \Closure $sector_definition = null){
        $current = resolve("dbFields");
        $items = $this->sectors[$source] ?? [];
        $new = static::init($this->current_model_obj,$items);
        
        app()->instance("dbFields",$new);
        
        if(!is_null($sector_definition)){
            $sector_definition($new);
        }
        
        $new->sort();
        if(!isset($this->sectors[$new_sector])){
            $this->sectors[$new_sector] = [];
        }
        foreach($new->toArray() as $sectorName=>$sectorFields){
            if($sectorName === "default") $sectorName =$new_sector;
            foreach($sectorFields as $key=>$field){
                $this->addField($field,$sectorName);
            }
        };
        
        if($current->throwaway){
            \Txd\Provider\FieldTypesServiceProvider::registerDbFields();
        }else{
            app()->instance("dbFields",$current);
        }
        return $this;
    }

    /**
     * Crea un settore unendo i campi definiti nei settori $sources.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     * 
     * @param string $sector
     * @param array|string $sources
     * 
     * @return $this
     */
    public function mergeSectors($sources = [],string $new_sector){
        if(is_string($sources)) $sources = [$sources];

        $items = $this->reduce($sources)->toArray("default");
        $new = static::init($this->current_model_obj,$items);
        
        $new->sort();
        if(!isset($this->sectors[$new_sector])){
            $this->sectors[$new_sector] = [];
        }
        foreach($new->toArray() as $sectorName=>$sectorFields){
            if($sectorName === "default") $sectorName =$new_sector;
            foreach($sectorFields as $key=>$field){
                $this->addField($field,$sectorName);
            }
        };
        
        return $this;
    }

    /**
     * setta il flag per l'inlcusione del settore specificato nei merge automatici. 
     * Se il flag è false, il settore non sarà incluso nei merge automatici (get_campi e get-campi_statici)
     * @param string $sector
     * @param bool $enable
     * 
     * @return $this
     */
    public function setSectorAutomaticMerge(string $sector, bool $enable){
        $this->skip_auto_merge_sectors[$sector] = $enable;
        return $this;
    }

    /**
     * Aggiunge un txdAttributes ai settori definiti in $sectors
     * @param \Txd\txdAttributes $field
     * @param array $sectors
     * 
     * @return void
     */
    public function addFieldToSectors(\Txd\txdAttributes $field,array $sectors){
        foreach($sectors as $sector){
            if(!isset($this->sectors[$sector])){
                $this->sectors[$sector] = [];
            }
            $this->sectors[$sector][$field->nome_campo] = $field;
        }
    }


    /**
     * sposta un txdAttributes presente nel db_field corrente dopo o prima di un altro campo esistente
     * 
     * @param \Txd\txdAttributes $field
     * @param string $fieldName
     * @param bool $before default false
     * 
     * @return void
     */
    public function moveField(\Txd\txdAttributes $field,string $fieldName,bool $before = false){
        foreach($this->sectors as $key => $sector){
            $found = array_search($field,$sector,true);
            if($found !== false){
                $found_ref = array_search($found,array_keys($sector));
                unset($sector[$found]);
                $key_ref = array_search($fieldName,array_keys($sector));
                if($key_ref !== false){
                    if(!$before) $key_ref++;
                    
                    $sector = array_slice($sector,0,$key_ref,true)+array($found=>$field)+array_slice($sector,$key_ref,count($sector),true);
                }else{
                    $sector = array_slice($sector,0,$found_ref,true)+array($found=>$field)+array_slice($sector,$found_ref,count($sector),true);
                }
                $this->sectors[$key] = $sector;
            }
        }
    }
 
    /**
     *  Rimuove un txdAttributes dai settori definiti in $sectors. se viene passato $all ) true il campo viene rimosso da tutti i settori
     * @param \Txd\txdAttributes $field
     * @param array $sectors
     * @param bool $all
     * 
     * @return void
     */
    public function removeFieldFromSectors(\Txd\txdAttributes $field,array $sectors,bool $all = false){
        if($all){
            $sectors = array_keys($this->sectors);
            unset($sectors["default"][$field->nome_campo]);
        }
        foreach($sectors as $sector){
            if(isset($this->sectors[$sector])){
                unset($this->sectors[$sector][$field->nome_campo]);
            }
        }
    }
}