<?php

namespace Txd\Traits\FieldTypes;

use Txd\FieldTypes\Enums\TXD_ATTRIBUTES_BUILD_MODE;
use Opis\Closure\SerializableClosure;
trait DrawableFieldTrait {
    public abstract function get_valore($raw = false);

    protected $custom_build_view = null;
    protected $view_is_cell = false;
    protected $custom_build_edit = null;
    
    public $hide_in_view_form = false;
    public $hide_in_edit_form = false;
    public $field_build_mode = TXD_ATTRIBUTES_BUILD_MODE::VIEW_AND_EDIT;
    
    public $skip_label = false;
    /**
     * determina se visualizzare o meno la label per il campo corrente. deve essere gestito nella blade del form
     * @param bool $value
     * 
     * @return $this
     */
    public function skip_label($value = true){
        $this->skip_label = $value == true;
        return $this;
    }

    /**
     * Genera l'HTML della modalità view del campo
     *
     * @return string
     */
    public function render_view(){
        if(!is_null($this->custom_build_view)){
            if($this->view_is_cell){
                return ($this->custom_build_view)($this->get_valore(),$this->current_model_obj);
            }else{
                return ($this->custom_build_view)($this);
            }
        }
        return $this->build_view();
    }
        
    /**
     * Genera l'HTML della modalità view del campo. metodo interno, nelle blade usare ->render_view()
     *  @internal
     * @return string
     */
    public function build_view(){
        return e($this->get_valore());
    }
    
    /**
     * assegna la closure per costruire la view del campo
     * @param \Closure $viewBuilder
     * *  signature: function($field):string
     * 
     * @return $this
     */
    public function custom_build_view(\Closure $viewBuilder){
        $this->custom_build_view = new SerializableClosure($viewBuilder);
        $this->view_is_cell = false;
        return $this;
    }

    


    /**
     * @param \Closure $cellView 
     *  signature: function($v,$r):string
     * 
     * accepts the field value ($v) and the model ($r). returns an HTML string
     * 
     * @return $this
     */
    public function cell(\Closure $cellView){
        $this->custom_build_view($cellView);
        $this->view_is_cell = true;
        return $this;
    }

    /**
     * stampa il campo in modalità edit
     * @return string
     */
    public function render_edit(){
        if(!is_null($this->custom_build_edit)){
            return ($this->custom_build_edit)($this);
        }
        return $this->build_edit();
    }
    /**
     * Genera l'HTML della modalità edit del campo. Metodo interno, nelle blade invocare ->render_edit()
     * @internal
     * @return string
     */
    public function build_edit(){
        return "Error: Missing type";
    }

    /**
     * assegna la closure per costruire la vista in edit del campo
     * @param \Closure $viewBuilder
     * *  signature: function($field):string
     * 
     * @return $this
     */
    public function custom_build_edit(callable $viewBuilder){
        $this->custom_build_edit = new SerializableClosure(function() use ($viewBuilder){return $viewBuilder($this);});
        return $this;
    }


    /**
     * Se impostato a true nasconde il campo (inclusa la label) dal form di edit
     * If true, the field will not be displayed in an edit form
     * 
     * @param bool $enable
     * 
     * @return $this
     */
    public function hide_from_edit(bool $enable = true){
        $this->hide_in_edit_form = $enable;
        return $this;
    }

    /**
     * Se impostato a true nasconde il campo (inclusa la label) dal form di view
     * If true, the field will not be displayed in an view form
     * 
     * @param bool $enable
     * 
     * @return $this
     */
    public function hide_from_view(bool $enable = true){
        $this->hide_in_view_form = $enable;
        return $this;
    }

    /**
     * Definisce la modalità di costruzione del campo nei form di view ed edit. 
     * I valori possibili sono TXD_ATTRIBUTES_BUILD_MODE::VIEW_AND_EDIT, TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY, TXD_ATTRIBUTES_BUILD_MODE::EDIT_ONLY</b> 
     * 
     * @param bool $txd_attribute_build_mode_flag
     * 
     * @return $this
     */

    public function set_build_mode(int $txd_attribute_build_mode_flag = 0){
        if(in_array($txd_attribute_build_mode_flag,[0,1,2])){
            $this->field_build_mode = $txd_attribute_build_mode_flag;
        }else{
            $this->field_build_mode = 0;
        }
        return $this;
    }

    /**
     * ritorna una stringa con le classi da applicare per la gestione della visibilità del campo in js
     * @return string
     */
    public function field_visibility_classes(){
        $view_class = $this->current_db_fields->view_visibility_class;
        $edit_class = $this->current_db_fields->edit_visibility_class;
        $out = "";
        if($this->hide_in_view_form) $out .= $edit_class;
        if($this->hide_in_edit_form) $out .= $view_class;
        return $out;
    }

    protected $cellClasses = [];
    protected $cellStyles = [];
    protected $cellAttributes = [];

    /**
     * assegna una classe alla cella del campo quando usato in txdGrid
     * @param string|\Closure|array $elements
     * 
     * @return $this
     */
    public function addCellClass($elements){
        if(!is_array($elements)){
            $elements = [$elements];
        }
        foreach($elements as $element){
            if(is_a($element,\Closure::class)){
                $element = new SerializableClosure($element);
            }   
            $this->cellClasses[] = $element;
        }
        return $this;
    }

    /**
     * assegna uno stile css alla cella del campo quando usato in txdGrid
     * @param mixed $key
     * @param string|\Closure $element
     * 
     * @return $this
     */
    public function addCellStyle($key,$element){
        if(is_a($element,\Closure::class)){
            $element = new SerializableClosure($element);
        }   
        $this->cellStyles[$key] = $element;
        return $this;
    }
    /**
     * assegna un attributo html alla cella del campo quando usato in txdGrid
     * @param mixed $key
     * @param string|\Closure $element
     * 
     * @return $this
     */
    public function addCellAttribute($key,$element){
        if(is_a($element,\Closure::class)){
            $element = new SerializableClosure($element);
        }   
        $this->cellAttributes[$key] = $element;
        return $this;
    }
    

    

    /**
     * restituisce un array contenente tutte le proprietà della cella del campo, formattato per essere usato in TxdJsonResource
     * @return array
     */
    public function getCellProperties(){
        $out["classes"] = implode(" ",array_map(function($class){
            if(is_a($class,SerializableClosure::class)){
                return $class($this);
            }
            return $class;
        },$this->cellClasses));
        $out["style"] = implode(";",collect($this->cellStyles)->map(function($style,$key){
            if(is_a($style,SerializableClosure::class)){
                return $key.":".$style($this);
            }
            return $key.":".$style;
        })->values()->toArray());
        $out["attributes"] = implode(" ",collect($this->cellAttributes)->map(function($attribute,$key){
            if(is_a($attribute,SerializableClosure::class)){
                return $key."=".$attribute($this);
            }
            return $key."=".$attribute;
        })->values()->toArray());

        return $out;
    }
}