<?php

namespace Txd\FieldTypes\Tests\Unit;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Session;
use Txd\FieldTypes\Enums\TXD_ATTRIBUTES_BUILD_MODE;
use Txd\FieldTypes\Tests\Moking\MockModel as MockModel;
use Txd\FieldTypes\Tests\TestCase;
use Txd\Traits\txdModel;
use Txd\txdFields;
use Watson\Validating\ValidatingTrait;

class TxdFieldsTest extends TestCase{
    public function testPhpunitEnv()
    {
        $this->assertTrue(true);
    }

    public function testMultipleSectors()
    {
        $modelClass = new class {
            use txdModel;
            
            public function db_fields_definition($db_fields)
            {
        
                $db_fields->AddText("nome");
                $db_fields->AddText("cognome");
                $db_fields->AddText("cellulare");
                $db_fields->addSelect("categoria_id");
                
                $db_fields->addSector("anagrafica",[
                    "nome",
                    "cognome",
                    "cellulare"
                ]);
                
                $db_fields->addSector("categorie",[
                    "categoria_id"
                ]);
                
                $db_fields->addSector("edit",[
                    "nome",
                    "cognome",
                    "cellulare",
                    "categoria_id"
                ]);
            }
        };
        
        
        $model = new $modelClass();
        $model->init_db_fields();
        
        $this->assertCount(3,$model->get_campi("anagrafica"));
        $this->assertCount(1,$model->get_campi("categorie"));
        $this->assertCount(4,$model->get_campi("edit"));
        
    }

    public function testDifferentImplementationBySector()
    {
        $modelClass = new class {
            use txdModel;
            
            public function db_fields_definition($db_fields)
            {
                $db_fields->AddText("nome");
                $db_fields->AddText("cognome");
                $db_fields->AddText("cellulare");
                $db_fields->addSelect("categoria_id");
                
                $db_fields->addSector("edit",[
                    "nome",
                    "cognome",
                    "cellulare",
                    "categoria_id"
                ]);
                
                // categoria_id viene rappresentato con un fieldType differente nel settore 'grid'
                $db_fields->addSector("grid",[
                    "nome",
                    "cellulare",
                    \Txd\FieldTypes\Text::new("categoria_id")
                ]);
            }
            
        };
        
        $model = new $modelClass();
        $model->init_db_fields();
        
        
        // se specifico il settore grid, categoria_id è di tipo Text
        $cat = $model->get_campi(["grid"])->retrieve("categoria_id");
        $this->assertEquals(\Txd\FieldTypes\Text::class,get_class($cat));
        
        $cat = $model->get_campi("default")->retrieve("categoria_id");
        // l'attributo è della tipologia definita inizialmente
        $this->assertEquals(\Txd\FieldTypes\Select::class,get_class($cat));
        
        // invocare get_campi senza parametri causa il merge di tutti i settori. Vince l'ultima definizione inserita.
        $cat = $model->get_campi()->retrieve("categoria_id");
        $this->assertEquals(\Txd\FieldTypes\Text::class,get_class($cat));
    }
    
    public function testFilteredInitDbFields()
    {
        $modelClass = new class {
            use txdModel;
            
            public function db_fields_definition($db_fields)
            {
                $db_fields->AddText("nome");
                $db_fields->AddText("cognome");
                $db_fields->AddText("cellulare");
                $db_fields->addSelect("categoria_id");
                
                $db_fields->addSector("edit",[
                    "nome",
                    "cognome",
                    "cellulare",
                    "categoria_id"
                ]);
                
                // categoria_id viene rappresentato con un fieldType differente nel settore 'grid'
                $db_fields->addSector("grid",[
                    "nome",
                    "cellulare",
                    \Txd\FieldTypes\Text::new("categoria_id")
                ]);
            }
            
        };
        
        $model = new $modelClass();
        $model->init_db_fields();
        $campi = $model->get_campi("edit"); // $campi è uguale a $campi_default
                
        $model->init_db_fields("edit");
        $campi_default = $model->get_campi(); // è stato inizializzato solo il settore 'edit', quindi $campi_default conterrà solo quelli
        
        // i due txdFields contengono gli stessi campi e nello stesso ordine
        $this->assertEquals($campi->toArray("default"),$campi_default->toArray("default"));
    }
    public function testChangeFieldDefinition()
    {
        $modelClass = new class {
            use txdModel;
            
            
            public function db_fields_definition($db_fields)
            {
                $db_fields->AddText("nome");
                $db_fields->AddText("cognome");
                $db_fields->AddText("cellulare");
                $db_fields->addSelect("categoria_id")->opzioni_select(["1"=>"Categoria 1","2"=>"Categoria 2"]);
                
                $db_fields->addSector("grid",[
                    "nome",
                    "cognome",
                    "cellulare",
                    "categoria_id"
                ]);
            }
            
        };
        
        Session::shouldReceive('has')
                    ->andReturn(false);
        
        $model = new $modelClass();
        $model->categoria_id = "1";
        $model->init_db_fields();
        $campi = $model->get_campi();
        $url = "test_url";
        //voglio aggiungere un link a categoria_id, ma nella grid base non deve esserci. lo aggiungo qui
        $campi->retrieve("categoria_id")
            ->cellWrap(function($v,$r) use($url){return "<a href='$url'>$v</a>";});
        
        // i due txdFields contengono gli stessi campi e nello stesso ordine
        $this->assertEquals("<a href='test_url'>Categoria 1</a>",$campi->retrieve("categoria_id")->render_view());
    }
    
    public function testTransformFields()
    {
        $db_fields = new txdFields();
        $db_fields->AddText("nome");
        $db_fields->AddText("cognome");
        $db_fields->AddText("cellulare");
        $db_fields->addSelect("categoria_id")->opzioni_select(["1"=>"Categoria 1","2"=>"Categoria 2"]);
        
        $db_fields->addSector("grid",[
            "nome",
            "cognome",
            "cellulare",
            "categoria_id"
        ]);
        
        foreach($db_fields as $field){
            $this->assertNotEquals(TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY,$field->field_build_mode);
        }
        $db_fields->transformFields(function($field){ return $field->set_build_mode(TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY);});
        
        foreach($db_fields as $field){
            $this->assertEquals(TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY,$field->field_build_mode);
        }
        
    }
    
    public function testSorting()
    {
        $modelClass = new class {
            use txdModel;
            
            
            public function db_fields_definition($db_fields)
            {
                $db_fields->AddText("nome");
                $db_fields->AddText("cognome");
                $db_fields->AddText("cellulare")->before("nome");
                $db_fields->AddText("categoria")->after("nome");
            }
            
        };
        
        $model = new $modelClass();
        $model->init_db_fields();
        $campi = $model->get_campi();
        $fieldsArray = [];
        foreach($campi as $field){
            $fieldsArray[] = $field->nome_campo;
        }
        $this->assertEquals(['cellulare','nome','categoria','cognome'],$fieldsArray);
    }
    
}