<?php

namespace Txd\Traits\FieldTypes;

use Txd\FieldTypes\Enums\TXD_ATTRIBUTES_BUILD_MODE;
use Txd\FieldTypes\Enums\TxdAttributeBuildMode;
use Txd\HtmlProperties;

trait DrawableFieldTrait
{
    abstract public function get_valore($raw = false);

    protected $custom_build_view = null;

    protected $view_is_cell = false;

    protected $custom_build_edit = null;

    public $hide_in_view_form = false;

    public $hide_in_edit_form = false;

    public $field_build_mode = TXD_ATTRIBUTES_BUILD_MODE::VIEW_AND_EDIT;

    public $skip_label = false;

    /**
     * determina se visualizzare o meno la label per il campo corrente. deve essere gestito nella blade del form
     *
     * @param  bool  $value
     * @return $this
     */
    public function skip_label($value = true)
    {
        $this->skip_label = $value == true;

        return $this;
    }

    /**
     * Genera l'HTML della modalità view del campo
     *
     * @return string
     */
    public function render_view()
    {
        if (!is_null($this->custom_build_view)) {
            return ($this->custom_build_view)($this);
        }

        return $this->build_view();
    }

    /**
     * Genera l'HTML della modalità view del campo. metodo interno, nelle blade usare ->render_view()
     *
     *  @internal
     *
     * @return string
     */
    public function build_view()
    {
        return e($this->get_valore());
    }

    /**
     * assegna la closure per costruire la view del campo
     *
     * @param  \Closure  $viewBuilder
     * *  signature: function($field):string
     * @return $this
     */
    public function custom_build_view(\Closure $viewBuilder)
    {

        $this->custom_build_view = new \Laravel\SerializableClosure\Serializers\Native($viewBuilder);
        $this->view_is_cell = false;

        return $this;
    }

    /**
     * @param  \Closure  $cellView
     *  signature: function($v,$r):string
     *
     * accepts the field value ($v) and the model ($r). returns an HTML string
     * @return $this
     */
    public function cell(\Closure $cellView)
    {
        $this->custom_build_view(fn ($el) => $cellView($el->get_valore(), $el->current_model_obj));
        $this->view_is_cell = true;

        return $this;
    }

    public function cellWrap(\Closure $wrapView)
    {
        $prev_cell = $this->custom_build_view;
        $this->custom_build_view(function ($el) use ($wrapView, $prev_cell) {
            if (is_null($prev_cell)) {
                return $wrapView($el->get_valore(), $el->current_model_obj, $el->build_view());
            }

            return $wrapView($el->get_valore(), $el->current_model_obj, $prev_cell($el));
        });

        return $this;
    }

    /**
     * stampa il campo in modalità edit
     *
     * @return string
     */
    public function render_edit()
    {
        if (!is_null($this->custom_build_edit)) {
            return ($this->custom_build_edit)($this);
        }

        return $this->build_edit();
    }

    /**
     * Genera l'HTML della modalità edit del campo. Metodo interno, nelle blade invocare ->render_edit()
     *
     * @internal
     *
     * @return string
     */
    public function build_edit()
    {
        return 'Error: Missing type';
    }

    /**
     * assegna la closure per costruire la vista in edit del campo
     *
     * @param  \Closure  $viewBuilder
     * *  signature: function($field):string
     * @return $this
     */
    public function custom_build_edit(callable $viewBuilder)
    {
        $this->custom_build_edit = new \Laravel\SerializableClosure\Serializers\Native(function () use ($viewBuilder) {
            return $viewBuilder($this);
        });

        return $this;
    }

    public function render_errors()
    {
        return "<div class='error-container'><template><span class='help-block'>__message__</span></template>".implode("\r\n", array_map(fn ($message) => "<span class='help-block'>$message</span>", $this->get_errors())).'</div>';
    }

    /**
     * Se impostato a true nasconde il campo (inclusa la label) dal form di edit
     * If true, the field will not be displayed in an edit form
     *
     *
     * @return $this
     */
    public function hide_from_edit(bool $enable = true)
    {
        $this->hide_in_edit_form = $enable;

        return $this;
    }

    /**
     * Se impostato a true nasconde il campo (inclusa la label) dal form di view
     * If true, the field will not be displayed in an view form
     *
     *
     * @return $this
     */
    public function hide_from_view(bool $enable = true)
    {
        $this->hide_in_view_form = $enable;

        return $this;
    }

    /**
     * Definisce la modalità di costruzione del campo nei form di view ed edit.
     * I valori possibili sono TXD_ATTRIBUTES_BUILD_MODE::VIEW_AND_EDIT, TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY, TXD_ATTRIBUTES_BUILD_MODE::EDIT_ONLY</b>
     *
     *
     * @return $this
     */
    public function set_build_mode(TxdAttributeBuildMode|int $txd_attribute_build_mode_flag = 0)
    {
        if (is_int($txd_attribute_build_mode_flag)) {
            $txd_attribute_build_mode_flag = TxdAttributeBuildMode::tryFrom($txd_attribute_build_mode_flag) ?? TxdAttributeBuildMode::VIEW_AND_EDIT;
        }
        $this->field_build_mode = $txd_attribute_build_mode_flag->value;

        return $this;
    }

    /**
     * ritorna una stringa con le classi da applicare per la gestione della visibilità del campo in js
     *
     * @return string
     */
    public function field_visibility_classes()
    {
        $view_class = $this->current_db_fields->view_visibility_class;
        $edit_class = $this->current_db_fields->edit_visibility_class;
        $out = '';
        if ($this->hide_in_view_form) {
            $out .= $edit_class;
        }
        if ($this->hide_in_edit_form) {
            $out .= $view_class;
        }

        return $out;
    }

    protected ?HtmlProperties $cellProperties = null;

    public function cellProperties(): HtmlProperties
    {
        if (!isset($this->cellProperties)) {
            $this->cellProperties = new HtmlProperties();
        }

        return $this->cellProperties;
    }
}
