<?php

namespace Txd;

use Txd\Traits\Extendable;
use Txd\Traits\FieldTypes\DrawableFieldTrait;
use Txd\Traits\Scopes\HasScope;

class txdAttributes
{
    use DrawableFieldTrait;
    use Extendable;
    use HasScope;

    // Validation
    public $skip_check_fillable = false;

    protected $model_rules = [];

    protected $model_type = null;

    // Shared
    public $nome_campo = '';

    public $tipo_campo = '';

    public $obbligatorio = false;

    protected $fillable = true;

    protected $disabled = null;

    protected $current_model_obj;

    /** @var \Txd\txdFields */
    protected $current_db_fields;

    public $prefix = null;

    // Render

    public $etichetta = '';

    protected $classi_html = [];

    public $placeholder = '';

    public $valore_fisso = null;

    public $parse_function = '';

    public $order = PHP_INT_MAX / 2;

    public $classi_etichetta = [];

    public $is_checked = null;

    public $classe_box = '';

    public $custom_html = '';

    protected $html_attributes = [];

    protected $campo_relazione = null;
    // Shared
    /**
     * @param  string  $nome
     * @param  string  $tipo
     * @param  string  $etichetta
     * @param  array  $classi_html
     */
    public function __construct($nome, $tipo = \Txd\FieldTypes\Text::class, $etichetta = '', $classi_html = [])
    {
        $this->nome_campo = $nome;
        $this->etichetta = strlen($etichetta) > 0 ? $etichetta : $this->etichetta_automatica($nome);
        $this->tipo_campo = $tipo;
        $this->classi_html = is_array($classi_html) ? $classi_html : [];
        $this->init_tipo();
    }

    public static function new(): static
    {
        return new static(...func_get_args());
    }

    // Render
    /**
     * ritorna il testo dell'etichetta (di default escaped per essere html-safe)
     *
     * @param  bool  $raw [default FALSE] se impostato a TRUE ritorna il valore di $etichetta cosi' com'e', altrimenti lo passa nella funzione e()
     * @return string
     */
    public function build_label_text($raw = false)
    {
        if ($raw) {
            return $this->etichetta;
        }

        return e($this->etichetta);
    }

    /**
     * @return $this
     */
    public function set_name_prefix(string $prefix)
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * Genera il nome del campo
     *
     * @param  string  $prefix prefisso al nome del campo (senza [])
     * @return string
     */
    public function build_nome_campo($prefix = null)
    {
        return $this->build_nome_input($prefix);
    }

    public function build_nome_input($prefix = null)
    {
        if ($this->current_model_obj->txdPrefixEnable ?? false) {
            if ($prefix != null) {
                $this->prefix = $prefix;
            } elseif (is_null($this->prefix)) {
                $classNameParts = explode('\\', get_class($this->current_model_obj));
                $this->prefix = $classNameParts[count($classNameParts) - 1];
            }
            $key = $this->current_model_obj->getKey();
            if (is_null($key)) {
                if (is_null($this->current_model_obj->_txdLocalPrefix)) {
                    $this->current_model_obj->_txdLocalPrefix = localUniqueString('new_', 4);
                }
                $key = $this->current_model_obj->_txdLocalPrefix;
            }

            return $this->prefix.'['.$key.']['.$this->nome_campo.']';
        }

        return $this->nome_campo;
    }

    public function get_nome_campo_sessione(): string
    {
        if ($this->current_model_obj->txdPrefixEnable ?? false) {
            return str_replace(']', '', str_replace('[', '.', $this->build_nome_campo()));
        } else {
            return $this->nome_campo;
        }

    }

    /**
     * genera l'etichetta a partire dal nome (rimuovendo eventualmente il suffisso _id)
     *
     * @param  string  $nome_campo
     * @return string
     */
    protected function etichetta_automatica($nome_campo)
    {

        $etichetta = $nome_campo;

        //se l'etichetta termina con _id togliamo in automatico il suffisso
        $parts = explode('_', $etichetta);

        if (count($parts) > 1 && $parts[count($parts) - 1] == 'id') {
            array_pop($parts);
        }

        return ucwords(implode(' ', $parts));
    }

    /**
     * imposta il modello di partenza all'interno del campo corrente
     *
     * @param  object  $obj
     */
    public function set_current_model_obj($obj)
    {

        $this->current_model_obj = $obj;
    }

    /**
     * imposta l'oggetto txdFields registrato internamente
     *
     * @param  \Txd\txdFields  $obj
     * @return void
     */
    public function set_current_db_fields(txdFields $obj)
    {

        $this->current_db_fields = $obj;
    }

    /**
     * restituisce l'oggetto txdFields registrato internamente
     *
     * @return \Txd\txdFields
     */
    public function get_current_db_fields()
    {

        return $this->current_db_fields;
    }

    public function get_current_model_obj()
    {

        return $this->current_model_obj;
    }

    /* ------------------------------------------------------------
     * SETTORI
     * ------------------- */

    /**
     * permette di impostare uno (o piu') settori
     *
     * @param  string|array  $nome
     * @return $this
     */
    public function add_settore($nome)
    {

        if (!is_array($nome)) {
            $nome = [$nome];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::addFieldToSectors($this, $nome);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->addFieldToSectors($this, $nome);
        }

        return $this;
    }

    /**
     * rimuove il settore passato dalla lista
     *
     * @param  string  $nome
     * @return $this
     */
    public function remove_settore($nome)
    {

        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::removeFieldFromSectors($this, [$nome]);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->removeFieldFromSectors($this, [$nome]);
        }

        return $this;
    }

    /**
     * svuota l'array dei settori per l'attributo in oggetto
     *
     * @return $this
     */
    public function clear_settori()
    {

        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::removeFieldFromSectors($this, [], true);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->removeFieldFromSectors($this, [], true);
        }

        return $this;
    }

    /* ------------------------------------------------------------
     * REGOLE
     * ------------------- */

    /**
     * permette di impostare una o piu' regole di validazione
     *
     * @param  string|array|callable|\Illuminate\Contracts\Validation\Rule  $rules nome della regola o array di regole. possono essere anche Closure o classi
     * @return $this
     */
    public function add_rule($rules)
    {

        if (is_string($rules)) {
            $rules = explode('|', $rules);
        }

        if (!is_array($rules)) {
            $rules = [$rules];
        }

        foreach ($rules as $rule) {
            if (array_search($rule, $this->model_rules, true) === false) {
                $this->model_rules[] = $rule;
            }
        }

        return $this;
    }

    /**
     * alias di add_rule($rules)
     *
     * @param  string|array|callable|\Illuminate\Contracts\Validation\Rule  $rules nome della regola o array di regole. possono essere anche Closure o classi
     * @return $this
     */
    public function add_rules($rules)
    {

        return $this->add_rule($rules);
    }

    /**
     * rimuove la regola passata come parametro dall'array
     *
     * @param  string|array|callable|\Illuminate\Contracts\Validation\Rule  $rules rule/rules da rimuovere
     * @return $this
     */
    public function remove_rule($rules)
    {

        if (is_string($rules)) {
            $rules = explode('|', $rules);
        }

        if (!is_array($rules)) {
            $rules = [$rules];
        }
        foreach ($rules as $rule) {
            if (($key = array_search($rule, $this->model_rules, true)) !== false) {
                unset($this->model_rules[$key]);
            }
        }

        return $this;
    }

    /**
     * svuota l'array delle regole
     *
     * @return $this
     */
    public function clear_rules()
    {

        $this->model_rules = [];

        return $this;
    }

    /**
     * sostituisce l'array delle regole con quello ricevuto
     *
     * @param  array  $arr_rules
     * @return $this
     */
    public function replace_rules($arr_rules = [])
    {

        $this->model_rules = $arr_rules;

        return $this;
    }

    /**
     * ritorna l'array delle regole per l'attributo corrente
     *
     * @internal
     *
     * @return array
     */
    public function get_model_rules()
    {

        return $this->model_rules;
    }

    /**
     * carica nell'attributo corrente le regole di validazione eventualmente definite a livello di modello
     *
     * @internal
     */
    public function parse_model_rules()
    {

        $tmp = new \Illuminate\Validation\ValidationRuleParser([]);
        $model_rules = [];

        if (isset($this->current_model_obj) && method_exists($this->current_model_obj, 'getRules')) {
            $model_rules = $this->current_model_obj->getRules();
        }

        $parsed_rules = json_decode(json_encode($tmp->explode($model_rules)), true);
        $parsed_rules_field = isset($parsed_rules['rules'][$this->nome_campo]) ? $parsed_rules['rules'][$this->nome_campo] : [];
        // ddd($parsed_rules_field);
        $this->add_rules($parsed_rules_field);
    }

    /* ------------------------------------------------------------
     * FILLABLE
     * ------------------- */

    /**
     * permette di impostare se un campo e' fillable o meno
     *
     * @param  bool  $enabled [default TRUE]
     * @return $this
     *
     * @throws \Exception
     */
    public function fillable($enabled = true)
    {

        if (!is_bool($enabled)) {
            throw new \Exception(__FUNCTION__.'() expected "boolean" as parameter, '.gettype($enabled).' given');
        }

        $this->fillable = $enabled;

        //se non e' stato ancora impostato il campo disabled, leghiamolo al comportamento di fillable
        if (is_null($this->disabled) && !$enabled) {
            $this->disabled = true;
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function isFillable()
    {

        return $this->fillable;
    }

    /* ------------------------------------------------------------
     * DISABLED
     * ------------------- */

    /**
     * permette di impostare se un campo e' disabled o meno.
     * di base questo comportamento segue la logica inversa di fillable, ma in questo modo
     * e' possibile specificare a mano un comportamento "disaccoppiato" delle due funzionalita'
     *
     * @param  bool  $enabled [default TRUE]
     * @return $this
     *
     * @throws \Exception
     */
    public function disabled($enabled = true)
    {

        if (!is_bool($enabled)) {
            throw new \Exception(__FUNCTION__.'() expected "boolean" as parameter, '.gettype($enabled).' given');
        }

        $this->disabled = $enabled;

        return $this;
    }

    /**
     * @return bool
     */
    public function isDisabled()
    {

        return $this->disabled ?? false;
    }

    /* ------------------------------------------------------------
     * Tipo (per conversione dati nel salvataggio dell'oggetto)
     * ------------------- */

    /**
     * imposta un tipo per l'attributo in modo che in fase di salvataggio il valore possa essere convertito
     * opportunamente per il DB
     *
     * @param  string  $tipo float | valuta | date | datetime
     *
     * @internal
     *
     * @return $this
     */
    public function set_model_type($tipo)
    {

        $this->model_type = $tipo;

        return $this;
    }

    /**
     * ritorna il tipo settato per l'attributo in oggetto (o null)
     *
     * @return string|null
     */
    public function get_model_type()
    {

        return $this->model_type;
    }

    /* ------------------------------------------------------------
     * CLASSI HTML
     * ------------------- */

    /**
     * aggiunge una (o piu' classi) html all'elenco
     *
     * @param  string|array  $classe nome della classe o array di classi
     * @return $this
     */
    public function add_classe_html($classe)
    {

        if (!is_array($classe)) {
            $classe = [$classe];
        }

        foreach ($classe as $item) {
            if (array_search($item, $this->classi_html, true) === false) {
                $this->classi_html[] = $item;
            }
        }

        return $this;
    }

    /**
     * rimuove la classe html passata dall'elenco
     *
     * @param  string  $classe
     * @return $this
     */
    public function remove_classe_html($classe)
    {

        if (($key = array_search($classe, $this->classi_html, true)) !== false) {
            unset($this->classi_html[$key]);
        }

        return $this;
    }

    /**
     * verifica se l'attributo ha la classe html passata o meno
     *
     * @param  string  $classe nome della classe
     * @return bool
     */
    public function hasHtmlClass($classe)
    {

        return array_search($classe, $this->classi_html, true) !== false;
    }

    /* ------------------------------------------------------------
     * ATTRIBUTI HTML
     * ------------------- */

    /**
     * Aggiunge un attributo html al campo. Non applica modifiche a campi già esistenti
     *
     * @param  string  $nome nome dell'attributo
     * @param  string  $valore valore dell'attributo
     * @return $this
     */
    public function add_html_attribute($nome, $valore = null)
    {

        if (array_search($nome, array_keys($this->html_attributes), true) === false) {
            $this->html_attributes[$nome] = $valore;
        }

        return $this;
    }

    /**
     * Aggiunge o modifica un attributo html sul campo
     *
     * @param  string  $nome nome dell'attributo
     * @param  string  $valore valore dell'attributo
     * @return $this
     */
    public function set_html_attribute($nome, $valore = null)
    {

        $this->html_attributes[$nome] = $valore;

        return $this;
    }

    /**
     * Imposta il placeholder del campo
     *
     *
     * @return $this
     */
    public function set_placeholder(string $valore)
    {

        $this->placeholder = $valore;

        return $this;
    }

    /**
     * rimuove l'attributo passato dall'elenco
     *
     * @param  string  $nome
     * @return $this
     */
    public function remove_html_attribute($nome)
    {

        unset($this->classi_html[$nome]);

        return $this;
    }

    /**
     * restituisce un array con gli attributi html dell'oggetto corrente
     *
     * @return array
     */
    public function html_attributes()
    {

        return $this->html_attributes;
    }

    /**
     * permette di impostare l'asterisco per i campi che non lo mostrano in automatico sulla base delle rules
     *
     * @param  bool  $param
     * @return $this
     */
    public function asterisco_manuale($param = true)
    {

        $this->obbligatorio = $param;

        return $this;
    }

    /**
     * Processa il valore ricevuto da database preparandolo per la stampa
     *
     * @param [type] $valore
     *
     * @internal
     *
     * @return void
     */
    public function parse_valore($valore)
    {
        return $valore;
    }

    /**
     * Parsing statico del valore secondo il FieldType usato per l'invocazione
     *
     * @param  any  $valore
     *
     * @internal
     *
     * @return any
     */
    public static function static_parse($valore)
    {
        $class = get_called_class();
        $instance = new $class('');

        return $instance->parse_valore($valore);
    }

    /**
     * ritorna il valore del campo corrente per l'oggetto impostato
     *
     * @param  bool  $raw [default: FALSE] se impostato a TRUE non effettua nessuna conversione
     * @return any
     */
    public function get_valore($raw = false)
    {
        $fullPath = explode(".",$this->nome_campo);
        $current = $this->current_model_obj;
        foreach($fullPath as $path){
            $current = optional($current)->$path ?? null;
        }
        
        $valore = $current;
        

        if (\Illuminate\Support\Facades\Session::has('errors')) {
            $valore = old($this->get_nome_campo_sessione());
        }

        if (!$raw) {
            $valore = $this->parse_valore($valore);
        }

        return $valore;
    }

    /**
     * ritorna la stringa con l'elenco delle classi html impostate sul campo in oggetto
     *
     * @internal
     *
     * @return string
     */
    public function get_classi_html()
    {
        return implode(' ', $this->classi_html);
    }

    /**
     * sulla base del tipo imposta alcune regole / informazioni standard (email, numeric, data_picker, etc...)
     *
     * @internal
     */
    public function init_tipo()
    {

    }

    /**
     * ritorna il tipo di campo da usare per la generazione dell'input nel form sulla base del tipo di campo
     *
     * @internal
     *
     * @return string
     */
    public function tipo_campo_form()
    {

        $tipo_campo_form = '';

        return $tipo_campo_form;
    }

    /**
     * verifica se il campo in oggetto ha la regola "required" nel set di regole passate come parametro
     * <br>oppure se e' stato instanziato con l'opzione obbligatorio a TRUE
     *
     * @return bool
     */
    public function is_mandatory()
    {

        return array_search('required', $this->model_rules) !== false || $this->obbligatorio;
    }

    /**
     * imposta un valore per l'ordinamento dei campi. il parametro Order prende la precedenza su tutte le altre soluzioini di ordinamento
     *
     * @return $this
     */
    public function set_order(int $order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * ordina l'attributo corrente dopo l'attributo $fieldName.
     * può essere filtrato su uno o più settori specifici. Se $sectors è un array vuoto, viene applicato su tutti i settori in cui sono presenti entrambi i campi
     *
     * @param  array|string  $sectors
     * @return $this
     */
    public function after(string $fieldName, $sectors = [])
    {
        if (is_string($sectors)) {
            $sectors = [$sectors];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::moveField($this, $fieldName, $sectors);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->moveField($this, $fieldName, $sectors);
        }

        return $this;
    }

    /**
     * ordina l'attributo corrente dopo l'attributo $fieldName.
     * può essere filtrato su uno o più settori specifici. Se $sectors è un array vuoto, viene applicato su tutti i settori in cui sono presenti entrambi i campi
     *
     * @param  array  $sectors
     * @return $this
     */
    public function before(string $fieldName, $sectors = [])
    {
        if (is_string($sectors)) {
            $sectors = [$sectors];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::moveField($this, $fieldName, $sectors, true);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->moveField($this, $fieldName, $sectors, true);
        }

        return $this;
    }

    protected function convertErrorNamesFormat($inputString)
    {
        // Replace the opening brackets with dots and remove the closing brackets
        $outputString = str_replace(['[', ']'], '.', $inputString);

        // Remove any double dots that may occur due to consecutive closing brackets
        $outputString = str_replace('..', '.', $outputString);

        // Remove any leading or trailing dots
        $outputString = trim($outputString, '.');

        return $outputString;
    }

    public function get_errors()
    {
        return optional(session()->get('errors'))->get($this->convertErrorNamesFormat($this->build_nome_campo())) ?? [];
    }

    public function has_errors()
    {
        return count($this->get_errors()) > 0;
    }
}
