<?php

namespace Txd;

use Closure;


class HtmlProperties
{
    protected array $_htmlClasses = [];

    protected array $_htmlStyles = [];

    protected array $_htmlAttributes = [];

    protected $_htmlId = null;
    /**
     * @param TKey $target
     */
    public function __construct(protected $target = null)
    {
        
    }

    /**
     * @return TKey
     */
    public function returnTarget(){
        return $this->target;
    }

    public function setHtmlId(string|callable $id)
    {
        $this->_htmlId = $id;
    }

    public function getHtmlId(): string|callable
    {
        return $this->_htmlId;
    }

    /* Classes */

    public function addHtmlClass(string|callable|array $classes)
    {
        if (!is_array($classes)) {
            $classes = [$classes];
        }
        $this->_htmlClasses = array_unique(array_merge($this->_htmlClasses, $classes));

        return $this;
    }

    public function removeHtmlClass(string|callable|array $classes)
    {
        if (!is_array($classes)) {
            $classes = [$classes];
        }
        $this->_htmlClasses = array_filter($this->_htmlClasses, fn ($class) => !in_array($class, $classes));

        return $this;
    }

    public function getHtmlClasses()
    {
        return $this->_htmlClasses;
    }

    public function setHtmlClasses(array $classes)
    {
        if (is_null($classes)) {
            $classes = [];
        }
        $this->_htmlClasses = $classes;

        return $this;
    }

    /* Styles */

    public function addHtmlStyle(string $key, string|callable $Style)
    {
        $this->_htmlStyles[$key] = $Style;

        return $this;
    }

    public function removeHtmlStyle(string $key)
    {
        if (array_key_exists($key, $this->_htmlStyles)) {
            unset($this->_htmlStyles[$key]);
        }

        return $this;
    }

    public function getHtmlStyles()
    {
        return $this->_htmlStyles;
    }

    public function setHtmlStyles(array $Styles)
    {
        if (is_null($Styles)) {
            $Styles = [];
        }
        $this->_htmlStyles = $Styles;

        return $this;
    }

    /* Attributes */

    public function addHtmlAttribute(string $key, null|string|callable $Attribute = null)
    {
        $this->_htmlAttributes[$key] = $Attribute;

        return $this;
    }

    public function removeHtmlAttribute(string $key)
    {
        if (array_key_exists($key, $this->_htmlAttributes)) {
            unset($this->_htmlAttributes[$key]);
        }

        return $this;
    }

    public function getHtmlAttributes()
    {
        return $this->_htmlAttributes;
    }

    public function setHtmlAttributes(array $Attributes)
    {
        if (is_null($Attributes)) {
            $Attributes = [];
        }
        $this->_htmlAttributes = $Attributes;

        return $this;
    }

    public function setHtmlProperties(array $classes = [], array $attributes = [], array $styles = [])
    {
        $this->setHtmlClasses($classes);
        $this->setHtmlAttributes($attributes);
        $this->setHtmlStyles($styles);

        return $this;
    }

    public function addHtmlProperties(array $classes = [], array $attributes = [], array $styles = [])
    {
        $this->addHtmlClass($classes);
        $this->setHtmlAttributes(array_merge($this->_htmlAttributes, $attributes));
        $this->setHtmlStyles(array_merge($this->_htmlStyles, $styles));

        return $this;
    }

    public function getHtmlProperties()
    {
        $properties = [
            'attributes' => $this->mapAttributes($this->_htmlAttributes),
            'classes' => $this->parseClasses($this->_htmlClasses),
            'styles' => $this->parseStyles($this->_htmlStyles),
        ];

        return $properties;
    }

    protected array $_renderArguments = [];

    public function setRenderArguments()
    {
        $this->_renderArguments = func_get_args();
    }

    protected function resolveValue($element): string
    {
        $value = $element;
        if (is_a($element, Closure::class)) {
            $value = $element(...$this->_renderArguments);
        }

        return (string) $value;
    }

    protected function mapAttributes($attrs)
    {
        $attributes = array_map(fn ($v) => $this->resolveValue($v), $attrs);
        $attributes = array_reduce(array_keys($attributes), function ($carry, $key) use ($attributes) {
            $carry[$key] = $attributes[$key];

            return $carry;
        }, []);

        return $attributes;
    }

    protected function mapClasses($classes): ?string
    {
        $output = implode(' ', $this->parseClasses($classes));

        return empty($output) ? null : $output;
    }

    protected function parseClasses(array $classes): array
    {
        return array_map(fn ($v) => $this->resolveValue($v), $classes);
    }

    public function getClassesString(): string
    {
        return $this->mapClasses($this->_htmlClasses);
    }

    public function getStylesString(): string
    {
        return $this->mapStyles($this->_htmlStyles);
    }

    protected function mapStyles($styles): ?string
    {
        $parsedStyles = $this->parseStyles($styles);

        $output = implode('; ', array_reduce(array_keys($parsedStyles), function ($carry, $key) use ($parsedStyles) {
            $carry[] = $key.': '.$parsedStyles[$key];

            return $carry;
        }, []));

        return empty($output) ? null : $output;
    }

    protected function parseStyles(array $styles): array
    {
        return array_map(fn ($v) => $this->resolveValue($v), $styles);
    }

    public static function to_html_attributes(array $attributes): string
    {
        $strings = [];
        foreach ($attributes as $name => $attribute) {
            $escaped = e($attribute);
            $strings[] = "$name='$escaped'";
        }

        return implode(' ', $strings);
    }

    public function renderHtmlProperties(): string
    {
        $attributes = array_merge(
            $this->mapAttributes($this->_htmlAttributes),
            [
                'class' => $this->renderClasses(),
                'style' => $this->renderStyle(),
                'id' => $this->renderId(),
            ]
        );

        return static::to_html_attributes(array_filter($attributes, fn ($val) => !is_null($val)));
    }

    public function renderId(): ?string
    {
        if (is_null($this->_htmlId)) {
            return null;
        }

        return $this->resolveValue($this->_htmlId);
    }

    public function renderClasses(): ?string
    {
        return $this->mapClasses($this->parseClasses($this->_htmlClasses));
    }

    public function renderStyle(): ?string
    {
        return $this->mapStyles($this->parseStyles($this->_htmlStyles));
    }

    public function renderAttributes(): string
    {
        return static::to_html_attributes($this->mapAttributes($this->_htmlAttributes));
    }
}
