<?php

namespace Txd;

use Exception;
use Txd\Traits\FieldTypes\DrawableFieldTrait;
use Txd\Traits\HtmlPropertiesTrait;
use Txd\Traits\Scopes\HasScope;

class BaseTxdAttributes
{
    use DrawableFieldTrait;
    use HasScope;
    use HtmlPropertiesTrait;
    
    protected $rules = [];

    // Shared
    public $nome_campo = '';



    /** @var \Txd\txdFields */
    protected $current_db_fields;

    public $prefix = null;

    // Render

    public string $etichetta = '';

    public $order = PHP_INT_MAX / 2;

    

    public $classe_box = '';

    public function __construct(string $nome, string $etichetta = '', array|string $classi_html = [])
    {
        $this->nome_campo = $nome;
        $this->etichetta = empty($etichetta) ? static::guessLabel($nome) : $etichetta;
        $this->htmlProperties()->addHtmlClass($classi_html);
    }

    public static function new(): static
    {
        return new static(...func_get_args());
    }

    // Render
    /**
     * ritorna il testo dell'etichetta (di default escaped per essere html-safe)
     *
     * @param  bool  $raw [default FALSE] se impostato a TRUE ritorna il valore di $etichetta cosi' com'e', altrimenti lo passa nella funzione e()
     * @return string
     */
    public function build_label_text($raw = false)
    {
        if ($raw) {
            return $this->etichetta;
        }

        return e($this->etichetta);
    }

    /**
     * @return $this
     */
    public function set_name_prefix(string $prefix)
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * Genera il nome del campo
     *
     * @param  string  $prefix prefisso al nome del campo (senza [])
     * @return string
     */
    public function build_nome_campo($prefix = null)
    {
        return $this->build_nome_input($prefix);
    }

    public function build_nome_input($prefix = null)
    {
        $output = $this->nome_campo;
        
        if(!empty($this->current_db_fields?->getModelPrefix($prefix))){
            $modelPrefix = $this->current_db_fields?->getModelPrefix($prefix);
            $output = "{$modelPrefix}[{$this->nome_campo}]";
        }      

        if(str_contains($output,".")){
            $re = '/\.([^\.]+)(?=[\.[])?/m';

            $subst = "[$1]";

            $output = preg_replace($re, $subst, $output);
        }
        
        return $output;
    }

    public function get_nome_campo_sessione(): string
    {
        return static::sessionNameFormat($this->build_nome_input());
    }

    public static function guessLabel($name){
        $etichetta = $name;

        //se l'etichetta termina con _id togliamo in automatico il suffisso
        $parts = explode('_', $etichetta);

        if (count($parts) > 1 && $parts[count($parts) - 1] == 'id') {
            array_pop($parts);
        }

        return ucwords(implode(' ', $parts));
    }
    

    

    /**
     * imposta l'oggetto txdFields registrato internamente
     *
     * @param  \Txd\txdFields  $obj
     * @return void
     */
    public function set_current_db_fields(txdFields $obj)
    {

        $this->current_db_fields = $obj;
    }

    /**
     * restituisce l'oggetto txdFields registrato internamente
     *
     * @return \Txd\txdFields
     */
    public function get_current_db_fields()
    {

        return $this->current_db_fields;
    }

    

    /* ------------------------------------------------------------
     * SETTORI
     * ------------------- */

    /**
     * permette di impostare uno (o piu') settori
     *
     * @param  string|array  $nome
     * @return $this
     */
    public function add_settore($nome)
    {

        if (!is_array($nome)) {
            $nome = [$nome];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::addFieldToSectors($this, $nome);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->addFieldToSectors($this, $nome);
        }

        return $this;
    }

    /**
     * rimuove il settore passato dalla lista
     *
     * @param  string  $nome
     * @return $this
     */
    public function remove_settore($nome)
    {

        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::removeFieldFromSectors($this, [$nome]);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->removeFieldFromSectors($this, [$nome]);
        }

        return $this;
    }

    /**
     * svuota l'array dei settori per l'attributo in oggetto
     *
     * @return $this
     */
    public function clear_settori()
    {

        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::removeFieldFromSectors($this, [], true);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->removeFieldFromSectors($this, [], true);
        }

        return $this;
    }

    /* ------------------------------------------------------------
     * REGOLE
     * ------------------- */

    /**
     * permette di impostare una o piu' regole di validazione
     *
     * @param  string|array|callable|\Illuminate\Contracts\Validation\Rule  $rules nome della regola o array di regole. possono essere anche Closure o classi
     * @return $this
     */
    public function addRule($rules)
    {
        $ruleParser = new \Illuminate\Validation\ValidationRuleParser([]);
        $explodedRules = $ruleParser->explode([$this->nome_campo => $rules]);
        

        $currentRules = $this->getRules();
        
        foreach ($explodedRules->rules[$this->nome_campo] as $rule) {
            if (array_search($rule, $currentRules, true) === false) {
                $this->rules[] = $rule;
            }
        }

        return $this;
    }


    /**
     * rimuove la regola passata come parametro dall'array
     *
     * @param  string|array|callable|\Illuminate\Contracts\Validation\Rule  $rules rule/rules da rimuovere
     * @return $this
     */
    public function removeRule($rules)
    {

        $ruleParser = new \Illuminate\Validation\ValidationRuleParser([]);
        $explodedRules = $ruleParser->explode([$this->nome_campo => $rules]);
        
        foreach ($explodedRules->rules[$this->nome_campo] as $rule) {
            if (($key = array_search($rule, $this->rules, true)) !== false) {
                unset($this->rules[$key]);
            }
        }

        return $this;
    }
    
   
    
    public function clearRules()
    {

        $this->rules = [];

        return $this;
    }

    
    protected $_model_rules = null;
    public function getModelRules($force = false){
        if(!$this->_model_rules){
            $rules = $this->current_db_fields?->getModelRules()[$this->nome_campo] ?? [];
            
            $this->_model_rules = $rules;
        }
        return $this->_model_rules;
    }
    
    /**
     *
     * @param boolean|null $includeModelRules overrides the internal txdAttribute state set with withoutModelRules()
     * @return void
     */
    public function getRules(?bool $includeModelRules = null){
        $rules = [];
        if($includeModelRules ?? !$this->skipModelRules){
            $rules = $this->getModelRules();
        }
        return array_merge($rules,$this->rules);
    }
    
    protected bool $skipModelRules = false;
    public function withoutModelRules(bool $enable = true){
        $this->skipModelRules = $enable;
        return $this;
    }

    

    /* ------------------------------------------------------------
     * DISABLED
     * ------------------- */

    /**
     * permette di impostare se un campo e' disabled o meno.
     * di base questo comportamento segue la logica inversa di fillable, ma in questo modo
     * e' possibile specificare a mano un comportamento "disaccoppiato" delle due funzionalita'
     */
    public function disabled(bool $enabled = true)
    {
        if($enabled){
            $this->htmlProperties()->addHtmlAttribute("disabled");
        }else{
            $this->htmlProperties()->removeHtmlAttribute("disabled");
        }
        
        return $this;
    }
    
    public function readOnly(bool $enabled = true)
    {
        if($enabled){
            $this->htmlProperties()->addHtmlAttribute("readonly");
        }else{
            $this->htmlProperties()->removeHtmlAttribute("readonly");
        }
        
        return $this;
    }

    
    
    /**
     * @return bool
     */
    public function excludedFromRules()
    {
        return array_key_exists("disabled",$this->htmlProperties()->getHtmlAttributes()) || array_key_exists("readonly",$this->htmlProperties()->getHtmlAttributes());
    }

    

    /**
     * Processa il valore ricevuto da modello preparandolo per la stampa
     *
     * @param [type] $valore
     *
     * @internal
     *
     * @return void
     */
    public function parse_valore($valore)
    {
        return $valore;
    }


    
    protected $_value = null;
    public function set_valore($valore){
        if($this->current_db_fields){
            throw new Exception("Cannot set a value if current_db_fields exists");
        }
        $this->_value = $valore;
    }
    
    /**
     * ritorna il valore del campo corrente per l'oggetto impostato
     *
     * @param  bool  $raw [default: FALSE] se impostato a TRUE non effettua nessuna conversione
     * @return any
     */
    public function get_valore($raw = false)
    {
        if($this->current_db_fields){
            $valore = $this->current_db_fields?->getModelValue($this->nome_campo);
        }else{
            $valore = $this->_value;
        }
        
        if (\Illuminate\Support\Facades\Session::has('errors')) {
            $valore = old($this->get_nome_campo_sessione());
        }

        if (!$raw) {
            $valore = $this->parse_valore($valore);
        }

        return $valore;
    }

    /**
     * ritorna il tipo di campo da usare per la generazione dell'input nel form sulla base del tipo di campo
     *
     * @internal
     *
     * @return string
     */
    public function tipo_campo_form()
    {

        $tipo_campo_form = '';

        return $tipo_campo_form;
    }

    /**
     * verifica se il campo in oggetto ha la regola "required" nel set di regole passate come parametro
     * <br>oppure se e' stato instanziato con l'opzione obbligatorio a TRUE
     *
     * @return bool
     */
    public function is_mandatory()
    {

        return array_search('required', $this->getRules()) !== false;
    }

    /**
     * imposta un valore per l'ordinamento dei campi. il parametro Order prende la precedenza su tutte le altre soluzioini di ordinamento
     *
     * @return $this
     */
    public function set_order(int $order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * ordina l'attributo corrente dopo l'attributo $fieldName.
     * può essere filtrato su uno o più settori specifici. Se $sectors è un array vuoto, viene applicato su tutti i settori in cui sono presenti entrambi i campi
     *
     * @param  array|string  $sectors
     * @return $this
     */
    public function after(string $fieldName, $sectors = [])
    {
        if (is_string($sectors)) {
            $sectors = [$sectors];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::moveField($this, $fieldName, $sectors);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->moveField($this, $fieldName, $sectors);
        }

        return $this;
    }

    /**
     * ordina l'attributo corrente dopo l'attributo $fieldName.
     * può essere filtrato su uno o più settori specifici. Se $sectors è un array vuoto, viene applicato su tutti i settori in cui sono presenti entrambi i campi
     *
     * @param  array  $sectors
     * @return $this
     */
    public function before(string $fieldName, $sectors = [])
    {
        if (is_string($sectors)) {
            $sectors = [$sectors];
        }
        if (!\Txd\Facades\DbFields::getInstance()->throwaway) {
            \Txd\Facades\DbFields::moveField($this, $fieldName, $sectors, true);
        } elseif (isset($this->current_db_fields)) {
            $this->current_db_fields->moveField($this, $fieldName, $sectors, true);
        }

        return $this;
    }

    protected static function sessionNameFormat($inputString)
    {
        return str_replace(']', '', str_replace('[', '.', $inputString));
    }

    public function get_errors()
    {
        return optional(session()->get('errors'))->get($this->get_nome_campo_sessione()) ?? [];
    }

    public function has_errors()
    {
        return count($this->get_errors()) > 0;
    }
}
