<?php

namespace Txd;

use Carbon\Carbon;
use Closure;
use Exception;
use Traversable;
use Txd\Facades\DbFields;
use Txd\FieldTypes\Hidden;
use Txd\FieldTypes\Text;

use WeakMap;

/**
 * Helper per definizione, costruzione e validazione di form complessi
 */
class txdFields implements \IteratorAggregate, \ArrayAccess
{
    public static $AUTO_APPLY_DB_FIELDS = true;

    

    public $throwaway;

    public $source;

    public $view_visibility_class = 'view_form_field';

    public $edit_visibility_class = 'edit_form_field';

    protected $sectors = [
        'default' => [],
    ];

    protected $skip_auto_merge_sectors = [];

    /**
     * @var null|object|\Txd\Contracts\TxdModel
     */
    protected null|object $current_model_obj;

    public function __construct(?object $current_model_obj = null, $throwaway = false)
    {
        $this->current_model_obj = $current_model_obj;
        $this->throwaway = $throwaway;
    }

    // implementazione arrayAccess e IteratorAggregate
    // funzioni per usare un oggetto txdFields come un array. tutti questi metodi funzionano solo sul settore "default"
    public function offsetSet($offset, $value): void
    {
        if (is_null($offset)) {
            $this->sectors['default'][] = $value;
        } else {
            $this->sectors['default'][$offset] = $value;
        }
    }

    public function offsetExists($offset): bool
    {
        return isset($this->sectors['default'][$offset]);
    }

    public function offsetUnset($offset): void
    {
        unset($this->sectors['default'][$offset]);
    }

    public function offsetGet($offset): mixed
    {
        return isset($this->sectors['default'][$offset]) ? $this->sectors['default'][$offset] : null;
    }

    public function getIterator(): Traversable
    {
        return new \ArrayIterator($this->sectors['default']);
    }
    // fine implementazione ArrayAccess e IteratorAggregate

    /**
     * inizializzazione statica di un oggetto txdFields passando direttamente l'array del settore "default"
     *
     */
    protected static function init(?object $current_model_obj, array $new_item_list)
    {
        $fields = new static($current_model_obj);
        foreach($new_item_list as $field){
            $fields->addField($field);
        }

        return $fields;
    }

    /**
     * ritorna un nuovo txdFields con il settore default popolato con il merge dei settori selezionati del txdfields corrente
     *
     * @param  array|string  $settori
     * @return \Txd\txdFields
     */
    public function reduce($settori = [], $keep = false)
    {
        if (is_string($settori)) {
            $settori = [$settori];
        }
        $new_item_list = [];

        if (count($settori) == 0) {
            $sectors_to_skip = array_keys(
                array_filter(
                    $this->skip_auto_merge_sectors,
                    function ($el) {
                        return !$el;
                    }
                )
            );

            $new_item_list = array_merge(
                ...array_values(
                    array_filter(
                        $this->sectors,
                        function ($key) use ($sectors_to_skip) {
                            return !in_array($key, $sectors_to_skip);
                        },
                        ARRAY_FILTER_USE_KEY)
                )
            );
        } else {
            foreach ($settori as $sector) {
                if (array_key_exists($sector, $this->sectors)) {
                    $new_item_list = array_merge($new_item_list, $this->sectors[$sector]);
                }
            }
        }

        $field = static::init($this->current_model_obj, $new_item_list);
        if ($keep) {
            foreach ($settori as $sector) {
                if ($this->sectors[$sector] ?? false) {
                    $field->sectors[$sector] = $this->sectors[$sector];
                }
            }
        }

        return $field;
    }

    /**
     * ritorna l'array dei settori con le loro proprietà
     *
     *
     * @return array
     */
    public function getSectors()
    {
        $out = [];
        foreach (array_keys($this->sectors) as $key) {
            $out[$key] = array_merge($this->sectorAttributes[$key] ?? [], ['automerge' => !($this->skip_auto_merge_sectors[$key] ?? false)]);
        }

        return $out;
    }

    public $sectorAttributes = [];

    /**
     * @param  mixed  $value
     * @return $this
     */
    public function setSectorAttribute(string $sector, string $key, $value)
    {
        if (!array_key_exists($sector, $this->sectorAttributes)) {
            $this->sectorAttributes[$sector] = [];
        }
        $this->sectorAttributes[$sector][$key] = $value;

        return $this;
    }

    protected static function uasort(array &$array, $value_compare_func)
    {
        $index = 0;
        foreach ($array as &$item) {
            $item = [$index++, $item];
        }

        $result = uasort($array, function ($a, $b) use ($value_compare_func) {
            $result = call_user_func($value_compare_func, $a[1], $b[1]);

            return $result == 0 ? $a[0] - $b[0] : $result;
        });

        foreach ($array as &$item) {
            $item = $item[1];
        }

        return $result;
    }

    /**
     * ordina i campi del settore selezionato secondo la funzione di sort passata. di default ordina per il campo ->order
     *
     * @param  string|null  $sector
     * @return $this
     */
    public function sort(callable $sorting_function = null, string $sector = 'default')
    {
        if (is_null($sorting_function)) {
            $sorting_function = function ($a, $b) {
                return $a->order - $b->order;

            };
        }

        static::uasort($this->sectors[$sector], $sorting_function);

        return $this;
    }

    /**
     * ritorna l'array del settore selezionato. Ritorna l'array dei settori in caso venga passato Null
     *
     *
     * @return array
     */
    public function toArray(string $sector = null)
    {
        if (is_null($sector)) {
            return $this->sectors;
        }

        return $this->sectors[$sector] ?? [];
    }

    /**
     * cerca la classe $typeName nei namespace disponibili per i FieldTypes
     */
    public static function findClass(string $typeName, array $additionalNamespaces = []): string
    {

        if (is_a($typeName,txdAttributes::class)) {
            return $typeName;
        }
        $namespaces = array_merge([config('txd_fieldtypes.local_fieldtypes_namespace', ''),'Txd\\FieldTypes' ], $additionalNamespaces);
        foreach ($namespaces as $namespace) {
            $class = $namespace.'\\'.$typeName;
            if (class_exists($class)) {
                return $class;
            }
        }
        
        if (config('app.debug', false)) {
            throw new Exception(__METHOD__." - FieldType '".$typeName."' is not a known php class");
        } else {
            \Log::warning(__METHOD__." - FieldType '".$typeName."' is not a known php class. '".Text::class."' used instead");
            $class = \Txd\FieldTypes\Text::class;
        }


        return $class;
    }

    /**
     * aggiunge un attributo all'elenco
     *
     * @param  string  $nome nome del campo su DB
     * @param  string  $tipo tipo di attributo. accetta il nome di una classe in Txd\FieldTypes, una classe custom o uno dei tipi legacy (text, email, password, data, select, etc...)
     * @param  string  $etichetta [default: ""] eventuale etichetta da stampare
     * @param  array  $classi_html elenco di classi da passare al campo
     * @return txdAttributes
     */
    public function add($nome, $typeName = "Txd\FieldTypes\Text", $etichetta = '', $classi_html = [])
    {

        

        $tmp = new $typeName($nome, $etichetta, $classi_html);

        return $this->addField($tmp);
    }
    
    public function __call($name, $args)
    {
        if (\Illuminate\Support\Str::startsWith($name, 'add')) {
            $className = substr($name, 3);
            
            $class = txdFields::findClass($className);
            

            return $this->add($args[0], $class, $args[1] ?? '', $args[2] ?? []);


        }


    }
    
    
    public function addRelated($nome, $finalModelClass, string $etichetta = '', string $settore = null)
    {
        $nameParts = explode('.', $nome);
        $nameParts = array_reverse($nameParts);
        
        $attribute = $finalModelClass::get_campi_statici($settore)->retrieve($nameParts[0]);
        if(is_null($attribute)){
            throw new Exception("addRelated was called on a field ($nameParts[0]) that does not have a fieldType on class $finalModelClass in sector $settore");
        }
        $attribute->nome_campo = $nome;
        if(strlen($etichetta)>0){
            $attribute->etichetta = $etichetta;
        }
        $attribute->disabled();

        return $this->addField($attribute);
    }

    /**
     * Rimuove il txdAttribute con nome $fieldName dal settore selezionato
     *
     * @return void
     */
    public function remove(string $fieldName, string $sector = 'default')
    {
        if (isset($this->sectors[$sector][$fieldName])) {
            unset($this->sectors[$sector][$fieldName]);
        }
    }

    /**
     * aggiunge il txdAttributes passato in ingresso ai settori selezionati (a default se non vengono selezionati settori)
     *
     * @param  \Txd\txdAttributes  $field
     * @param  array|string  $sectors
     * @return $field
     */
    public function addField(txdAttributes $field, $sectors = [])
    {
        if (is_string($sectors)) {
            $sectors = [$sectors];
        }
        
        $field->set_current_db_fields($this);

        if (count($sectors) == 0) {
            $this->sectors['default'][$field->nome_campo] = $field;
        } else {
            $this->addFieldToSectors($field, $sectors);
        }

        return $field;
    }

    /**
     * ritorna il txdAttribute con nome del campo corrispondente, cercandolo nel settore segnalato (default se null)
     *
     * @param  string  $sector
     * @return \txd\txdAttributes
     */
    public function retrieve(string $fieldName, string $sector = null)
    {

        if (is_null($sector)) {
            $sector = 'default';
        }

        return $this->sectors[$sector][$fieldName] ?? null;
    }

    /**
     * imposta il modello di partenza all'interno del campo corrente
     */
    public function set_current_model_obj(?object $obj)
    {

        $this->current_model_obj = $obj;
    }

    /**
     * restituisce il modello a cui il campo è assegnato
     */
    public function current_model_obj()
    {

        return $this->current_model_obj;
    }

    /**
     * attributo per abilitare la stampa del campo di verifica. deve essere gestito nella blade
     *
     * @var bool
     */
    public $skip_verification_field = false;

    /**
     * crea il campo di verifica per ricostruzione univoca del txdFields per la validazione
     *
     * @return string
     */
    public function verification_field($settori = null)
    {
        return static::static_verification_field($settori ?? $this->source);
    }

    public static function static_verification_field($settori = [])
    {
        $random = localUniqueString(null, 20);
        
        session()->flash('_old_input.txd_verification', $random);
        $hidden = new Hidden('txd_verification');
        $hidden->set_valore($random);
        $sessionStore = static::getVerificationTokens();
        $sessionStore[$random] = ['source' => $settori, 'exp' => Carbon::now()->addMinutes(config('txd_fieldtypes.txd_verification_expiration_interval', 15))];
        static::setVerificationTokens($sessionStore);

        return $hidden->render_edit();
    }

    /**
     * recupera l'elenco di token di verifica validi dalla sessione
     *
     * @return array
     */
    public static function getVerificationTokens()
    {
        $now = Carbon::now();

        return array_filter((array) session()->get('txdFieldsVerification'), function ($el) use ($now) {
            return $el['exp'] > $now;
        });
    }

    /**
     * cerca nell'array di token di verifica il token passato. Se lo trova, il token viene invalidato e ritorna l'array di settori corrispondente. Se $keep è a true, il token non viene invalidato al primo controllo (per supporto finestre modali)
     *
     * @param  string  $token
     * @param  bool  $keep
     * @return array|null
     */
    public static function getVerificationTokenMatch(?string $token, $keep = false)
    {
        $sessionStore = txdFields::getVerificationTokens();

        if (!$keep) {
            $tokenMatch = \Illuminate\Support\Arr::pull($sessionStore, $token);
        } else {
            $tokenMatch = $sessionStore[$token] ?? null;
        }

        txdFields::setVerificationTokens($sessionStore);
        if (!is_null($tokenMatch)) {
            return $tokenMatch['source'] ?? null;
        }

        return null;
    }

    /**
     * salva in sessione l'array di token di verifica validi
     *
     * @param  array  $tokensArray
     * @return void
     */
    public static function setVerificationTokens($tokensArray)
    {
        session()->put('txdFieldsVerification', $tokensArray);
    }

    /**
     * ritorna l'istanza corrente (utile per Facade)
     *
     * @return $this
     */
    public function getInstance()
    {
        return $this;
    }

    /**
     * Aggiunge un settore identificato da $sector. Il settore viene costruito eseguendo la funzione $sector_definition.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti.
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     *
     * @param  \Closure  $sector_definition
     * @return $this
     */
    public function addSector(string $sector, array|Closure $sector_definition, bool $add_to_parent_sector = false)
    {
        if (is_array($sector_definition)) {
            foreach ($sector_definition as $field) {

                if (is_string($field)) {
                    optional($this->retrieve($field))->add_settore($sector);
                } else {
                    $this->addField($field, [$sector]);
                }
            }

            return $this;
        }
        $new = new txdFields($this->current_model_obj);
        DbFields::pushInstance($new);

        $sector_definition($new);
        $new->sort();
        if (!isset($this->sectors[$sector])) {
            $this->sectors[$sector] = [];
        }
        foreach ($new->toArray() as $sectorName => $sectorFields) {
            // if($sectorName === "default"){
            //     $sectorName =$sector;
            //     if($add_to_parent_sector){
            //         foreach($sectorFields as $key=>$field){
            //             $this->addField($field,"default");
            //         }
            //     }
            // }
            $sectors = [$sectorName];
            if ($sectorName === 'default') {
                $sectors = [$sector];
                if ($add_to_parent_sector) {
                    $sectors = ['default', $sector];
                }
            }
            foreach ($sectorFields as $key => $field) {
                $this->addField($field, $sectors);
            }
        }
        DbFields::popInstance();

        return $this;
    }

    /**
     * elimina il settore indicato
     *
     *
     * @return $this
     */
    public function deleteSector(string $sector)
    {
        if (isset($this->sectors[$sector])) {
            unset($this->sectors[$sector]);
        }

        return $this;
    }

    /**
     * modifica il settore identificato da $sector. Il settore viene passato alla funzione $sector_definition per essere modificato.
     * il contenuto precedente del settore viene completamente sostituito dalle nuove definizioni.
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     *
     * @return $this
     */
    public function editSector(string $sector, Closure $sector_definition)
    {

        $new = static::init($this->current_model_obj, $this->toArray($sector));
        DbFields::pushInstance($new);

        $sector_definition($new);
        $new->sort();

        if (isset($this->sectors[$sector])) {
            unset($this->sectors[$sector]);
        }

        $this->sectors[$sector] = [];
        foreach ($new->toArray() as $sectorName => $sectorFields) {
            if ($sectorName === 'default') {
                $sectorName = $sector;
            }
            foreach ($sectorFields as $key => $field) {
                $this->addField($field, $sectorName);
            }
        }
        DbFields::popInstance();

        return $this;
    }

    /**
     * Copia il contenuto del settore $source in un settore identificato da $sector. Il settore viene costruito eseguendo la funzione $sector_definition.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti.
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     *
     * @param  string  $sector
     * @return $this
     */
    public function cloneSector(string $source, string $new_sector, Closure $sector_definition = null)
    {
        $current = resolve('dbFields');
        $items = $this->sectors[$source] ?? [];
        $new = static::init($this->current_model_obj, $items);

        DbFields::pushInstance($new);

        if (!is_null($sector_definition)) {
            $sector_definition($new);
        }

        $new->sort();
        if (!isset($this->sectors[$new_sector])) {
            $this->sectors[$new_sector] = [];
        }
        foreach ($new->toArray() as $sectorName => $sectorFields) {
            if ($sectorName === 'default') {
                $sectorName = $new_sector;
            }
            foreach ($sectorFields as $key => $field) {
                $this->addField($field, $sectorName);
            }
        }
        DbFields::popInstance();

        return $this;
    }

    /**
     * Crea un settore unendo i campi definiti nei settori $sources.
     * nel caso in cui il settore $sector fosse già esistente, i nuovi campi vengono aggiunti ai campi preesistenti
     * i campi vengono ordinati secondo il campo $order prima di essere aggiunti
     *
     * @param  string  $sector
     * @param  array|string  $sources
     * @return $this
     */
    public function mergeSectors($sources, string $new_sector)
    {
        if (is_string($sources)) {
            $sources = [$sources];
        }

        $items = $this->reduce($sources)->toArray('default');
        $new = static::init($this->current_model_obj, $items);

        $new->sort();
        if (!isset($this->sectors[$new_sector])) {
            $this->sectors[$new_sector] = [];
        }
        foreach ($new->toArray() as $sectorName => $sectorFields) {
            if ($sectorName === 'default') {
                $sectorName = $new_sector;
            }
            foreach ($sectorFields as $key => $field) {
                $this->addField($field, $sectorName);
            }
        }

        return $this;
    }

    /**
     * setta il flag per l'inlcusione del settore specificato nei merge automatici.
     * Se il flag è false, il settore non sarà incluso nei merge automatici (get_campi e get-campi_statici)
     *
     * @return $this
     */
    public function setSectorAutomaticMerge(string $sector, bool $enable)
    {
        $this->skip_auto_merge_sectors[$sector] = $enable;

        return $this;
    }

    /**
     * Aggiunge un txdAttributes ai settori definiti in $sectors
     *
     * @param  \Txd\txdAttributes  $field
     */
    public function addFieldToSectors(txdAttributes $field, array $sectors): void
    {
        foreach ($sectors as $sector) {
            if (!isset($this->sectors[$sector])) {
                $this->sectors[$sector] = [];
            }
            $this->sectors[$sector][$field->nome_campo] = $field;
        }
    }

    /**
     * sposta un txdAttributes presente nel db_field corrente dopo o prima di un altro campo esistente nei settori selezionati. se $sectors è vuoto viene applicato su tutti i settori
     *
     * @param  \Txd\txdAttributes  $field
     * @param  bool  $before default false
     */
    public function moveField(txdAttributes $field, string $fieldName, array $sectors = [], bool $before = false): void
    {
        if (count($sectors) === 0) {
            $sectors = array_keys($this->sectors);
        }
        foreach ($this->sectors as $key => $sector) {
            if (!in_array($key, $sectors)) {
                continue;
            }
            $found = array_search($field, $sector, true);
            if ($found !== false) {
                $found_ref = array_search($found, array_keys($sector));
                unset($sector[$found]);
                $key_ref = array_search($fieldName, array_keys($sector));
                if ($key_ref !== false) {
                    if (!$before) {
                        $key_ref++;
                    }

                    $sector = array_slice($sector, 0, $key_ref, true) + [$found => $field] + array_slice($sector, $key_ref, count($sector), true);
                } else {
                    $sector = array_slice($sector, 0, $found_ref, true) + [$found => $field] + array_slice($sector, $found_ref, count($sector), true);
                }
                $this->sectors[$key] = $sector;
            }
        }
    }

    /**
     *  Rimuove un txdAttributes dai settori definiti in $sectors. se viene passato $all ) true il campo viene rimosso da tutti i settori
     *
     * @param  \Txd\txdAttributes  $field
     */
    public function removeFieldFromSectors(txdAttributes $field, array $sectors, bool $all = false): void
    {
        if ($all) {
            $sectors = array_keys($this->sectors);
            unset($sectors['default'][$field->nome_campo]);
        }
        foreach ($sectors as $sector) {
            if (isset($this->sectors[$sector])) {
                unset($this->sectors[$sector][$field->nome_campo]);
            }
        }
    }

    /**
     * applica la stessa trasformazione ($callback) a tutti i campi dei settori selezionati
     *
     * @param  array  $settori
     * @return $this
     */
    public function transformFields(Closure $callback, $settori = [], $all_sectors = false)
    {
        if ($all_sectors) {
            $settori = array_keys($this->sectors);
        }

        if (is_string($settori)) {
            $settori = [$settori];
        }
        if (count($settori) === 0) {
            $settori = ['default'];
        }

        foreach ($settori as $settore) {
            foreach ($this->sectors[$settore] as $field) {
                $callback($field);
            }
        }

        return $this;
    }

    /**
     * filtra i campi secondo l'esito di $callback. La modifica viene effettuata in place
     *
     * @param  array  $settori
     * @param  bool  $all_sectors
     * @return $this
     */
    public function filter(Closure $callback, $settori = [], $all_sectors = false)
    {
        if ($all_sectors) {
            $settori = array_keys($this->sectors);
        }

        if (is_string($settori)) {
            $settori = [$settori];
        }
        if (count($settori) === 0) {
            $settori = ['default'];
        }

        foreach ($settori as $settore) {
            $fields = array_keys($this->sectors[$settore]);
            foreach ($fields as $field_key) {
                $field = $this->sectors[$settore][$field_key];
                if (!$callback($field)) {
                    $this->remove($field_key, $settore);
                }
            }
        }

        return $this;
    }

    public function getRules()
    {
        $mergedRules = array_map(function (txdAttributes $field) {
            return $field->excludedFromRules()?null: $field->getRules();
        }, iterator_to_array($this));
        
        return array_filter($mergedRules,fn($v) => !is_null($v));
    }
    
    protected bool $skipModelRules = false;
    public function withoutModelRules(bool $enable = true){
        $this->skipModelRules = $enable;
        return $this;
    }
    
    public function getModelRules(){
        if($this->skipModelRules) return [];
        return $this->current_model_obj()?->getRules() ?? [];
    }
        
    
    public function getModelValue(string $fieldName){
        $fullPath = explode(".",$fieldName);
        $current = $this->current_model_obj;
        foreach($fullPath as $path){
            $current = $current->{$path} ?? null;
        }
        return $current;
    }
    
    public function getModelPrefix(?string $prefix = null) : string {
        if ($this->current_model_obj?->txdPrefixEnable ?? false) {    
            if (is_null($prefix)) {
                $classNameParts = explode('\\', get_class($this->current_model_obj));
                $prefix = $classNameParts[count($classNameParts) - 1];
            }
            
            $key = $this->current_model_obj?->getKey();
            if (is_null($key)) {
                $key = static::localPrefix($this->current_model_obj);
            }
            return $prefix.'['.$key.']';
        }
        return "";
    }
    
    protected static WeakMap $localPrefixes;
    
    public static function localPrefix($object,$value = null){
        if(!isset(static::$localPrefixes)){
            static::$localPrefixes = new WeakMap();
        }
        if(!is_null($value)){
            static::$localPrefixes->offsetSet($object,$value);
        }
        if(!static::$localPrefixes->offsetExists($object)){
            static::$localPrefixes->offsetSet($object,localUniqueString('new_', 4));
        }
        return static::$localPrefixes->offsetGet($object);
    }
    
    
    
    public function getFieldLabels()
    {
        return array_map(function (txdAttributes $field) {
            return $field->build_label_text();
        }, iterator_to_array($this));
    }

    public function makeValidator($data = [])
    {
        return \Illuminate\Support\Facades\Validator::make($data, $this->getRules(), [], $this->getFieldLabels());
    }
}
