<?php

namespace Txd\Forms;

use Illuminate\Support\Collection;
use Txd\FieldTypes\Enums\TXD_ATTRIBUTES_BUILD_MODE;
use Opis\Closure\SerializableClosure;
use Txd\Http\Resources\TxdJsonResource;
use Txd\txdAttributes;

class TxdGrid extends TxdQueryForm {

    /**
     * @var \Illuminate\Http\Resources\Json\AnonymousResourceCollection
     */
    public $resource;
    

    public function __construct(\Txd\txdFields $base = null,$defaults = null)
    {
        parent::__construct($base,$defaults ?? $base->current_model_obj());
        $this->htmlAttributes = [
            "class" => "txd-grid"
        ];
        $this->transformFields(function(txdAttributes $field){
            $field->set_build_mode(TXD_ATTRIBUTES_BUILD_MODE::VIEW_ONLY);
        });
        $base->loadModelAttributes();
        $base->current_model_obj()->fill(request()->input());
        $this->view = "txd::Forms.Grids.default";
    }

    public $paginationInSession = true;
    public $sessionName = null;
    protected function getCurrentPage(){
        $maxPage = 1;
        if($this->perPage>0){
            $maxPage = ceil($this->query->count()/$this->perPage);
        }
        

        if(!$this->paginationInSession){
            return min($maxPage,$this->page??request()->input($this->pageName,null));
        }

        $session_key = config("txd_forms.session_path").".pagination.".($this->sessionName??urlencode(request()->url())).".".$this->pageName;
        $p = null;
        if(!is_null($this->page)){
            $p = min($maxPage,$this->page);
        }else if(request()->has($this->pageName)){
            $p = min($maxPage,request()->input($this->pageName,null));
        }else{
            $p = min($maxPage,session()->get($session_key,null));
        }
        if(!is_null($p)) session()->put($session_key,$p);
        return session()->get($session_key,null);
    }
    

    protected $row_index = 0;

    public $viewTotalRows = "txd::Forms.Grids.defaultTotalRows";
    public $viewRow = "txd::Forms.Grids.defaultRow";
    public $viewHeader = "txd::Forms.Grids.defaultHeader";
    public $viewPagination = "txd::Forms.Grids.defaultPagination";

    protected function defaultViewData(){
        return [
            "grid"=>$this,
            "TotalRowsView"=>$this->viewTotalRows,
            "RowView"=>$this->viewRow,
            "HeaderView"=>$this->viewHeader,
            "PaginationView"=>$this->viewPagination,
        ];
    }

    public $orderBy = [];

    protected function sortQuery(){
        $this->prepareOrderByValues();
        if(is_a($this->query,Collection::class) && count($this->orderBy)>0){
            // implementazione valida da Laravel 8 in poi
            // $this->query = $this->query->sortBy($this->orderBy);
            // fallback per laravel 6, solo il primo livello di sort viene applicato in quanto i metodi sort di php non sono consistenti
            [$key,$direction] = $this->orderBy[0];
            $this->query = $this->query->sortBy($key,SORT_REGULAR,strtolower($direction)==="desc");

        }else{
            foreach($this->orderBy as $field){
                [$columnLabel,$direction] = $field;
                $column = $this->sortableFields[$columnLabel] ?? null;
                if(!is_null($column)){
                    $this->query = $this->query->orderBy($column,$direction ?? "asc");
                }
            }
        }
        return $this->query;
    }

    public $perPage = null;
    public $page = null;
    public $pageName = "page";
    
    public function paginate($perPage = null,$pageName ="page"){
        if(is_null($perPage)){
            $perPage = 20;
        }
        $this->perPage = $perPage;
        $this->pageName = $pageName;
        return $this;
    }


    public function build(){
        $this->query = $this->sortQuery($this->query);
        if(is_null($this->perPage)){
            $this->perPage = $this->query->count();
        }
        $this->store_final_query();
        $this->query = $this->query->paginate($this->perPage,['*'],$this->pageName,$this->getCurrentPage());
        
        $this->appliedToQuery = true;
        $this->row_index = 0;
        $this->resource = TxdJsonResource::scopedFieldObject($this,function(){return TxdJsonResource::collection($this->query);});

    }

    
    public function printPagination(){
        return $this->resource->resource->toHtml();
    }

    public function totalRows(){
        return $this->resource->resource->total();
    }


    protected $rowClasses = [];
    protected $rowStyles = [];
    protected $rowAttributes = [];

    /**
     * assegna una classe alle righe della grid
     * @param string|\Closure|array $elements
     * 
     * @return $this
     */
    public function addRowClass($elements){
        if(!is_array($elements)){
            $elements = [$elements];
        }
        foreach($elements as $element){
            if(is_a($element,\Closure::class)){
                $element = new SerializableClosure($element);
            }   
            $this->rowClasses[] = $element;
        }
        return $this;
    }

    /**
     * assegna uno stile css alle righe della grid
     * @param mixed $key
     * @param string|\Closure $element
     * 
     * @return $this
     */
    public function addRowStyle($key,$element){
        if(is_a($element,\Closure::class)){
            $element = new SerializableClosure($element);
        }   
        $this->rowStyles[$key] = $element;
        return $this;
    }

    /**
     * assegna un attributo html alle righe della grid
     * @param mixed $key
     * @param string|\Closure $element
     * 
     * @return $this
     */
    public function addRowAttribute($key,$element){
        if(is_a($element,\Closure::class)){
            $element = new SerializableClosure($element);
        }   
        $this->rowAttributes[$key] = $element;
        return $this;
    }

    /**
     * restituisce un array contenente tutte le proprietà delle row
     * @return array
     */
    public function getRowProperties(){
        $index = $this->row_index;
        $this->row_index++;
        $out = ["fields"=>[]];
        
        $out["classes"] = implode(" ",array_map(function($class) use($index){
            if(is_a($class,SerializableClosure::class)){
                return $class($this->current_model_obj,$index);
            }
            return $class;
        },$this->rowClasses));
        $out["style"] = implode(";",collect($this->rowStyles)->map(function($style,$key) use($index){
            if(is_a($style,SerializableClosure::class)){
                return $key.":".$style($this->current_model_obj,$index);
            }
            return $key.":".$style;
        })->values()->toArray());
        $out["attributes"] = implode(" ",collect($this->rowAttributes)->map(function($attribute,$key) use($index){
            if(is_a($attribute,SerializableClosure::class)){
                return $key."=".$attribute($this->current_model_obj,$index);
            }
            return $key."=".$attribute;
        })->values()->toArray());

        return $out;
    }
    
    protected $sortableFields = [];
    /**
     * assegna le colonne su cui è possibile ordinare la grid
     * @param array $fields
     * 
     * @return $this
     */
    public function sortables($fields = null){
        if(is_null($fields)) return $this->sortableFields;
        //TODO: rivedere questo workaround...
        $processed_values = array_merge(array_combine(array_values($fields),array_values($fields)),$fields);
        $this->sortableFields = array_intersect_key($processed_values,$this->toArray("default"));
        return $this;
    }

    public $orderName = "order";
    protected function prepareOrderByValues(){
        $direction = "asc";
        if(request()->has($this->orderName."Desc")){
            $direction = "desc";
        }
        $field = request()->input($this->orderName."Asc",request()->input($this->orderName."Desc",null));
        
        if(!is_null($field) && array_key_exists($field,$this->sortableFields)){
            $this->orderBy = array_merge([[$field,$direction]],$this->orderBy);
        }else{
            $field = null;
        }
        
        if($this->paginationInSession){
            $session_key = config("txd_forms.session_path").".pagination.".($this->sessionName??urlencode(request()->url())).".orderBy";
            $sessionvalues = session()->get($session_key,[]);
            $filteredSessionValues = array_filter($sessionvalues,function($el) use($field){
                return $el[0]!==$field;
            });
            $this->orderBy = array_merge($this->orderBy,$filteredSessionValues);
            session()->put($session_key,$this->orderBy);
        }
    }
    

    
}