<?php

namespace Txd\GotenbergClient\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Log;
use Response;

class GotenbergPdfService
{
    protected $client;

    public function __construct()
    {
        $this->client = new Client();
    }

    /**
     * riceve una serie di stringhe come parametri e ritorna la versione concatenata con un solo "/" tra ogni parte
     *
     * @param string $baseurl stringa di partenza
     * @param string ...$parts elenco di parti da concatenare
     */
    private static function buildSlashPath(string $baseurl, string ...$parts): string
    {
        $baseurl = rtrim($baseurl, '/');
        array_walk($parts, function (&$part)
        {
            $part = trim($part, '/');
        });

        return $baseurl.'/'.join('/', $parts);
    }

    /**
     * verifica se il paramentro APP_ENV e' tra quelli che supportano il lancio di alcune eccezioni che in produzione sarebbero saltate
     *
     * @return bool
     */
    private static function throwable_environment()
    {
        return in_array(config('app.env'), config('par.exception_env', []));
    }

    /**
     * ritorna una stringa (o un array) con alcune info utili per il debug
     *
     * @return string|array
     */
    private static function context_error_details($as_json = true)
    {
        $out = [];
        try {
            $out = array_filter([
                'url' => request()->fullUrl(),
                'input' => request()->except(['password', 'password_confirmation']),
                'userId' => auth()->id(),
                'email' => auth()->user() ? auth()->user()->email : null,
                'ip' => request()->ip(),
            ]);
        } catch (\Throwable $e) {
            $out = [];
        }

        if ($as_json) {
            return json_encode($out);
        } else {
            return $out;
        }
    }

    /**
     * ritorna il PDF generato a partire da un contenuto HTML chiamando il servizio Gotenberg
     *
     * @param string $htmlContent il contenuto HTML da convertire in PDF
     * @param bool $stream_as_pdf [default: true] se true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string
     * @param array $gotenbergOptions opzioni specifiche di gotenberg
     * @param string|null $title [default: null] titolo del PDF
     *
     * @return Response|string|bool se $stream_as_pdf è true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string, in caso di errore ritorna false
     */
    public function convertHtmlToPdf(string $htmlContent, bool $stream_as_pdf = true, array $gotenbergOptions = [], ?string $title = null)
    {
        $default_options = collect([
            [
                'name' => 'files',
                'contents' => $htmlContent,
                'filename' => 'index.html',
            ],
            [
                'name' => 'paperWidth',
                'contents' => '210mm', //A4
            ],
            [
                'name' => 'paperHeight',
                'contents' => '297mm', //A4
            ],
        ])->keyBy('name');

        $override = collect($gotenbergOptions)->keyBy('name');

        $full_options = $default_options->merge($override)->values()->toArray();

        $pdfContent = false;
        try {
            $response = $this->client->post(static::buildSlashPath(config('gotenberg_pdf.endpoint'), '/forms/chromium/convert/html'), [
                'multipart' => $full_options,
            ]);

            $pdfContent = $response->getBody()->getContents();
        } catch (RequestException $e) {
            if (static::throwable_environment()) {
                throw $e;
            }
            Log::error(__METHOD__, ['eccezione' => $e, 'contesto' => static::context_error_details(false)]);
            $pdfContent = false;
        }

        if ($pdfContent && $stream_as_pdf) {
            return Response::make($pdfContent, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline;'.($title ? 'filename="'.$title.'"' : ''),
            ]);
        }

        return $pdfContent;
    }

    /**
     * ritorna il PDF generato a partire da un URL chiamando il servizio Gotenberg
     *
     * @param string $url l'URL da convertire in PDF
     * @param bool $stream_as_pdf [default: true] se true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string
     * @param array|null $extraHttpHeaders [default: null] eventuali header HTTP aggiuntivi da inviare al servizio Gotenberg
     * @param string|null $title [default: null] titolo del PDF
     *
     * @return Response|string|bool se $stream_as_pdf è true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string, in caso di errore ritorna false
     */
    public function convertUrlToPdf(string $url, bool $stream_as_pdf = true, ?array $extraHttpHeaders = [], ?string $title = null)
    {
        $pdfContent = false;
        try {
            $response = $this->client->post(static::buildSlashPath(config('gotenberg_pdf.endpoint'), '/forms/chromium/convert/url'), [
                'multipart' => [
                    [
                        'name' => 'url',
                        'contents' => $url,
                    ],
                    [
                        'name' => 'userAgent',
                        'contents' => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko)',
                    ],
                    [
                        'name' => 'extraHttpHeaders',
                        'contents' => json_encode($extraHttpHeaders),
                    ],
                ],
            ]);

            $pdfContent = $response->getBody()->getContents();
        } catch (RequestException $e) {
            if (static::throwable_environment()) {
                throw $e;
            }
            Log::error(__METHOD__, ['eccezione' => $e, 'contesto' => static::context_error_details(false)]);
            $pdfContent = false;
        }

        if ($pdfContent && $stream_as_pdf) {
            return Response::make($pdfContent, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline;'.($title ? 'filename="'.$title.'"' : ''),
            ]);
        }

        return $pdfContent;
    }


    /**
     * ritorna il PDF merged a partire da un array di file chiamando il servizio Gotenberg
     *
     * @param array $files array con all'interno i file
     * @param bool $stream_as_pdf [default: true] se true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string
     * @param array|null $extraHttpHeaders [default: null] eventuali header HTTP aggiuntivi da inviare al servizio Gotenberg
     * @param string|null $title [default: null] titolo del PDF
     *
     * @return Response|string|bool se $stream_as_pdf è true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string, in caso di errore ritorna false
     */
    public function mergeFiles($files, $stream_as_pdf, ?string $title = null){
        $pdfContent = false;

        $options = [
            'multipart' => [
                [
                    'name' => 'userAgent',
                    'contents' => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko)',
                ],
            ],
        ];

        foreach ($files as  $key =>  $value) {
            $options['multipart'][] = [
                'name' =>  'files',
                'contents' =>  $value,
                'filename' => $key
            ];
        }
        try {
            $path  = static::buildSlashPath(config('gotenberg_pdf.endpoint'), '/forms/pdfengines/merge');
            $response = $this->client->post($path, $options);

            $pdfContent = $response->getBody()->getContents();
        } catch (RequestException $e) {
            if (static::throwable_environment()) {
                throw $e;
            }
             Log::error(__METHOD__, ['eccezione' => $e, 'contesto' => static::context_error_details(false)]);
            $pdfContent = false;
        }

        if ($pdfContent && $stream_as_pdf) {
            return Response::make($pdfContent, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline;'.($title ? 'filename="'.$title.'"' : ''),
            ]);
        }

        return $pdfContent;
    }

    /**
     * ritorna il PDF generato a partire da uno o più file di tipo Word, Excel, PowerPoint, ODT, RTF, TXT, CSV, HTML chiamando il servizio Gotenberg
     *
     * @param array $files array con all'interno i file da convertire in PDF. Ogni elemento dell'array può essere:
     *                      - una istanza di \Illuminate\Http\UploadedFile (es. da una request)
     *                      - una istanza di \SplFileInfo
     *                      - una stringa con il path assoluto di un file esistente
     *                      - un array associativo con chiavi "name" (nome del file con estensione) e "contents" (contenuto del file come stringa o stream)
     * @param bool $stream_as_pdf se true ritorna una Response, altrimenti ritorna il contenuto del PDF come raw string
     * @param string|null $title
     * @return void
     */
    public function convertWordToPdf(array $files, bool $stream_as_pdf, ?string $title = null)
    {
        $pdfContent = false;

        $multipart = [];

        if (count($files) > 1) {
            $multipart[] = [
                'name'     => 'merge',
                'contents' => 'true',
            ];
        }

        foreach ($files as $key => $value) {
            $filename = null;
            $contents = null;

            // 1) Laravel UploadedFile
            if ($value instanceof \Illuminate\Http\UploadedFile) {
                $filename = $value->getClientOriginalName() ?: $value->getClientOriginalName();
                $contents = fopen($value->getRealPath(), 'r');
            }
            // 2) SplFileInfo
            elseif ($value instanceof \SplFileInfo) {
                $filename = $value->getFilename();
                $contents = fopen($value->getRealPath(), 'r');
            }
            // 3) Plain filesystem path
            elseif (is_string($value) && is_file($value)) {
                $filename = basename($value);
                $contents = fopen($value, 'r');
            }
            // 4) Associative ["name" => "file.docx", "contents" => "...bytes..."]
            elseif (is_array($value) && isset($value['name'], $value['contents'])) {
                $filename = $value['name'];
                $contents = $value['contents']; // can be string or stream
            }
            // 5) Key => value where key is filename and value is raw contents
            elseif (is_string($key) && $key !== '' && isset($value)) {
                $filename = $key;
                $contents = $value; // assume raw bytes or stream
            }

            // validate filename and extension
            if (!$filename || !preg_match('/\.(docx|doc|odt|rtf|txt|csv|pptx|ppt|xlsx|xls|html?)$/i', $filename)) {
                // Gotenberg needs a recognizable extension
                throw new \InvalidArgumentException("Each part must include a valid filename with extension (got '{$filename}').");
            }

            $multipart[] = [
                'name'     => 'files',
                'contents' => $contents,
                'filename' => $filename,
            ];
        }

        $options = [
            'multipart' => $multipart,
            'headers' => ['Accept' => 'application/pdf'], // optional
        ];

        try {
            $path  = static::buildSlashPath(config('gotenberg_pdf.endpoint'), '/forms/libreoffice/convert');
            $response = $this->client->post($path, $options);
            $pdfContent = $response->getBody()->getContents();
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            if (static::throwable_environment()) {
                throw $e;
            }
            \Log::error(__METHOD__, ['eccezione' => $e, 'contesto' => static::context_error_details(false)]);
            $pdfContent = false;
        }

        if ($pdfContent && $stream_as_pdf) {
            return \Response::make($pdfContent, 200, [
                'Content-Type'        => 'application/pdf',
                'Content-Disposition' => 'inline;'.($title ? 'filename="'.$title.'"' : ''),
            ]);
        }

        return $pdfContent;
    }
}
