<?php

namespace Txd\JobsMonitor\Commands;

use Carbon\Carbon;
use DB;
use Exception;
use Illuminate\Console\Command;

class JobsQueueMonitorCommand extends Command
{
    protected $signature = 'txd:jobs-monitor';

    protected $description = 'Monitor jobs queue and notify on trigger';

    protected $current_default_db_connection;

    public function __construct()
    {
        parent::__construct();

        $this->current_default_db_connection = config('database.default');
    }

    /**
     * @return \Illuminate\Console\View\Components\Factory|self
     */
    protected function console_write()
    {
        if (property_exists($this, 'components') && is_a($this->{'components'}, 'Illuminate\Console\View\Components\Factory')) {
            /** @var \Illuminate\Console\View\Components\Factory $callable_message_component */
            $callable_message_component = $this->{'components'};
        } else {
            /** @var self $callable_message_component */
            $callable_message_component = $this;
        }

        return $callable_message_component;
    }

    public function handle()
    {
        if (! config('txd_jobs_monitor.enabled')) {
            $this->console_write()->info('Jobs monitor is disabled');

            return;
        }

        $this->console_write()->line('Monitoring jobs queue');

        $notify_alert_up = false;
        $notification_messages = [];

        // Config values
        $minutes = config('txd_jobs_monitor.time_range_min', 5);
        $maxCountPerQueue = config('txd_jobs_monitor.max_jobs_per_queue');
        $maxTotalCount = config('txd_jobs_monitor.max_jobs_queue_len');

        $sqliteDbPath = config('txd_jobs_monitor.sqlite_db_path');

        // Set up the SQLite connection using the configured path
        config(['database.connections.custom_sqlite' => [
            'driver' => 'sqlite',
            'database' => $sqliteDbPath,
            'prefix' => '',
        ]]);

        // Temporarily use the custom SQLite connection
        DB::setDefaultConnection('custom_sqlite');

        initJobsMonitorStatsDB(false);

        $last_notify_up = optional(DB::table('notifications')->where('action', 'notify_up')->first())->executed ?? false;
        $last_notify_down = optional(DB::table('notifications')->where('action', 'notify_down')->first())->executed ?? false;

        $this->storeCurrentStatus();

        $queueCounts = DB::table('queue_job_status')
            ->select('queue_name', DB::raw('avg(job_count) as job_count'))
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            // ->where('job_count', '>', ($maxCountPerQueue ?? 0))
            ->groupBy('queue_name')
            ->get()
            ->keyBy('queue_name')
            ->map(function ($queue_info) {
                return $queue_info->job_count;
            })
            ->toArray();

        $totalJobsPerQueue = DB::table('queue_job_status')
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            ->get()
            ->groupBy('queue_name')
            ->toArray();

        $totalJobs = DB::table('queue_job_status')
            ->select('queue_name', DB::raw('avg(job_count) as total_jobs'))
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            ->groupBy('queue_name')
            ->get()
            ->map(function ($queue_info) {
                return $queue_info->total_jobs;
            })
            ->toArray();

        $totalJobs = array_sum($totalJobs);
        // dd($queueCounts, $totalJobs);

        if ($maxTotalCount !== null && $totalJobs > $maxTotalCount) {
            $notification_messages[] = 'Total jobs count [average '.number_format($totalJobs, 2, ',', '.').'] exceeded the limit of '.number_format($maxTotalCount, 0, '', '.').' jobs';
            $notify_alert_up = true;
        } else {
            $notification_messages[] = 'Total jobs count [average '.number_format($totalJobs, 2, ',', '.').'] is under the trigger value of '.number_format($maxTotalCount, 0, '', '.').' jobs';
        }

        if ($maxCountPerQueue !== null) {
            foreach ($queueCounts as $queue => $avgCount) {
                $avgCount = round((float) $avgCount, 2);
                if ($avgCount > (float) $maxCountPerQueue) {
                    $notification_messages[] = "Queue {$queue} exceeded its limit of ".$maxCountPerQueue.' jobs [average: '.number_format($avgCount, 2, ',', '.').']';
                    $notify_alert_up = true;
                } else {
                    $notification_messages[] = "Queue {$queue} is under its trigger value of {$maxCountPerQueue} jobs [average: ".number_format($avgCount, 2, ',', '.').']';
                }
            }
        }

        $triggered = false;
        $this->console_write()->info(implode("\n", $notification_messages));
        // dd($queueCounts);

        if ($notify_alert_up && ! $last_notify_up) {
            DB::table('notifications')->updateOrInsert(['action' => 'notify_up'], ['executed' => true]);
            DB::table('notifications')->updateOrInsert(['action' => 'notify_down'], ['executed' => false]);
            $triggered = $this->triggerAlert('TRIGGER_UP', implode("\n", $notification_messages), $totalJobsPerQueue);
        }

        if (! $notify_alert_up && ! $last_notify_down) {
            DB::table('notifications')->updateOrInsert(['action' => 'notify_up'], ['executed' => false]);
            DB::table('notifications')->updateOrInsert(['action' => 'notify_down'], ['executed' => true]);
            $triggered = $this->triggerAlert('TRIGGER_DOWN', implode("\n", $notification_messages), $totalJobsPerQueue);
        }

        if (! $triggered) {
            $this->console_write()->warn('No need to trigger alert');
        }

        $this->console_write()->line('Deleting old records');
        DB::table('queue_job_status')
            ->where('created_at', '<', Carbon::now()->subMinutes($minutes * 10))
            ->delete();

        DB::setDefaultConnection($this->current_default_db_connection);
    }

    /**
     * Store current status in SQLite DB.
     */
    private function storeCurrentStatus()
    {
        $timestamp = Carbon::now();

        switch (config('queue.default')) {
            case 'database':
                $jobs_per_queue = DB::connection($this->current_default_db_connection)->table('jobs')
                    ->select('queue', DB::connection($this->current_default_db_connection)->raw('count(*) as jobs_count'))
                    ->groupBy('queue')
                    ->get()
                    ->keyBy('queue')
                    ->map(function ($queue_info) {
                        return $queue_info->jobs_count;
                    })
                    ->toArray();
                break;

            case 'redis': //TODO: Implement Redis driver
            default:
                throw new Exception('Queue driver not implemented: '.config('queue.default'));
                break;
        }

        foreach ($jobs_per_queue as $queue => $count) {
            DB::table('queue_job_status')->insert([
                'queue_name' => $queue,
                'job_count' => $count,
                'created_at' => $timestamp,
            ]);
        }
    }

    /**
     * Trigger an alert based on the exceeded condition.
     */
    private function triggerAlert(string $level, string $message, $queueCounts = null)
    {
        switch ($level) {
            case 'TRIGGER_UP':
                $this->console_write()->error($message);
                break;
            case 'TRIGGER_DOWN':
                $this->console_write()->warn($message);
                break;
            default:
                $this->console_write()->error($level.' not recognized');
                $this->console_write()->line($message);
                break;

        }
        foreach (config('txd_jobs_monitor.active_notification') as $notification => $enabled) {
            if ($enabled) {
                app('Txd\JobsMonitor\Notifications\\'.ucfirst($notification))
                    ->setNotificationLevel($level)
                    ->setInfo($message)
                    ->setQueueCounts($queueCounts)
                    ->send();
            }
        }

        return true;
    }
}
