<?php

namespace Txd\JobsMonitor\Notifications;

use Exception;
use Mail;
use Txd\JobsMonitor\Contracts\NotificationContract;

class Email implements NotificationContract
{
    protected $config;

    protected $notificationLevel;

    protected $info;

    protected $queueCounts;

    protected $totalJobs;

    public function __construct()
    {
        $this->config = config('txd_jobs_monitor.notifications.email');
    }

    public function setNotificationLevel(string $notificationLevel): NotificationContract
    {
        $this->notificationLevel = $notificationLevel;

        return $this;
    }

    public function setInfo(string $info): NotificationContract
    {
        $this->info = $info;

        return $this;
    }

    public function setQueueCounts(array $queueCounts): NotificationContract
    {
        $this->queueCounts = $queueCounts;

        return $this;
    }

    public function setTotalJobs(int $totalJobs): NotificationContract
    {
        $this->totalJobs = $totalJobs;

        return $this;
    }

    public function send(): bool
    {
        $recipients = $this->getRecipients();

        if (empty($recipients)) {
            throw new Exception(__METHOD__.' | No valid recipients found');
        }

        switch ($this->notificationLevel) {
            case 'TRIGGER_DOWN':
                $testo = 'Il numero di JOBS in coda sul sistema '.config('app.name').' è sceso al di sotto della soglia di trigger';
                break;
            case 'TRIGGER_UP':
                $testo = '<b>ATTENZIONE!</b> Il numero di JOBS in coda sul sistema '.config('app.name').' è salito al di sopra della soglia di trigger';
                break;
            default:
                return false;
        }

        $testo .= '<p>'.nl2br($this->info);

        $testo .= '<p>JOBS per queue:';
        $tot_jobs = 0;
        foreach ($this->queueCounts as $queue => $info) {
            $testo .= '<p><b>'.$queue.'</b>:';
            foreach ($info as $value) {
                $testo .= '<br>'.$value->created_at.': '.$value->job_count;
                $tot_jobs += $value->job_count;
            }
        }

        $testo .= '<p><b>Totale JOBS:</b> '.$tot_jobs;

        Mail::to($recipients)->send(new \Txd\JobsMonitor\Mail\NotificationMail(
            $this->subjectFromLevel($this->notificationLevel),
            $testo
        ));

        return true;
    }

    protected function subjectFromLevel($level): string
    {
        switch ($level) {
            case 'TRIGGER_DOWN':
                $subject = config('app.name').' | Jobs Anomalies Notification - Under Quota';
                break;
            case 'TRIGGER_UP':
                $subject = config('app.name').' | Jobs Anomalies Notification - Over Quota';
                break;
            default:
                $subject = config('app.name').' | Jobs Anomalies Notification';
                break;
        }

        return $subject;
    }

    protected function getRecipients()
    {
        return collect(explode(',', $this->config['recipients']))
            ->map(function ($value) {
                return trim($value);
            })
            ->filter(function ($value) {
                return filter_var($value, FILTER_VALIDATE_EMAIL);
            })->toArray();
    }
}
