<?php

namespace Txd\JobsMonitor\Commands;

use Carbon\Carbon;
use DB;
use Illuminate\Console\Command;
use Txd\JobsMonitor\Classes\JobsQueueMonitor;

class JobsQueueMonitorCommand extends Command
{
    /** @var JobsQueueMonitor */
    protected $jobsQueueMonitor;

    protected $signature = 'txd:jobs-monitor';

    protected $description = 'Monitor jobs queue and notify on trigger';

    public function __construct(JobsQueueMonitor $jobsQueueMonitor)
    {
        parent::__construct();

        $this->jobsQueueMonitor = $jobsQueueMonitor;
    }

    /**
     * @return \Illuminate\Console\View\Components\Factory|self
     */
    protected function console_write()
    {
        if (property_exists($this, 'components') && is_a($this->{'components'}, 'Illuminate\Console\View\Components\Factory')) {
            /** @var \Illuminate\Console\View\Components\Factory $callable_message_component */
            $callable_message_component = $this->{'components'};
        } else {
            /** @var self $callable_message_component */
            $callable_message_component = $this;
        }

        return $callable_message_component;
    }

    public function handle()
    {
        if (! config('txd_jobs_monitor.enabled')) {
            $this->console_write()->info('Jobs monitor is disabled');

            return;
        }

        $this->console_write()->info('Monitoring jobs queue');

        $notify_alert_up = false;
        $notification_messages = [];

        // Config values
        $minutes = config('txd_jobs_monitor.time_range_min', 5);
        $maxCountPerQueue = config('txd_jobs_monitor.max_jobs_per_queue');
        $maxTotalCount = config('txd_jobs_monitor.max_jobs_queue_len');

        $this->jobsQueueMonitor->initJobsMonitorStatsDB(false);

        $this->jobsQueueMonitor->storeCurrentStatus();

        $queueCounts = $this->jobsQueueMonitor->queueCounts($minutes);
        $recap_notification_range = $minutes * 3;
        $totalJobsPerQueue = $this->jobsQueueMonitor->jobsPerQueue($recap_notification_range);
        $totalJobs = $this->jobsQueueMonitor->totalJobs($minutes);

        if ($maxTotalCount !== null) {
            [
                'status' => $status,
                'messages' => $notification_messages
            ] = $this->jobsQueueMonitor->evalTotalJobs($totalJobs, $maxTotalCount);

            if ($status) {
                $notify_alert_up = true;
            }
        }

        if ($maxCountPerQueue !== null) {
            [
                'status' => $status,
                'messages' => $notification_messages
            ] = $this->jobsQueueMonitor->evalQueueTrigger($queueCounts, $maxCountPerQueue);

            if ($status) {
                $notify_alert_up = true;
            }
        }

        $this->console_write()->info(implode("\n", $notification_messages));

        [
            'trigger_up' => $trigger_up,
            'trigger_down' => $trigger_down
        ] = $this->jobsQueueMonitor->evalAndSetTriggers($notify_alert_up);

        if ($trigger_up) {
            $this->triggerAlert('TRIGGER_UP', implode("\n", $notification_messages), $totalJobsPerQueue, $recap_notification_range);
        }

        if ($trigger_down) {
            $this->triggerAlert('TRIGGER_DOWN', implode("\n", $notification_messages), $totalJobsPerQueue, $recap_notification_range);
        }

        if (! $trigger_up && ! $trigger_down) {
            $this->console_write()->warn('No need to trigger alert');
        }

        $this->console_write()->info('Deleting old records...');
        DB::connection('txd_jobs_sqlite')->table('queue_job_status')
            ->where('created_at', '<', Carbon::now()->subMinutes($minutes * 10))
            ->delete();

    }

    /**
     * Trigger an alert based on the exceeded condition.
     */
    private function triggerAlert(string $level, string $message, $queueCounts = null, ?int $recap_notification_range = null)
    {
        $this->console_write()->info('Triggering alert: '.$level);

        switch ($level) {
            case 'TRIGGER_UP':
                $this->console_write()->warn($message);
                break;
            case 'TRIGGER_DOWN':
                $this->console_write()->warn($message);
                break;
            default:
                $this->console_write()->error($level.' not recognized');
                $this->console_write()->info($message);
                break;

        }
        foreach (config('txd_jobs_monitor.active_notification') as $notification => $enabled) {
            if ($enabled) {
                try {
                    app('Txd\JobsMonitor\Notifications\\'.ucfirst($notification))
                        ->setNotificationLevel($level)
                        ->setInfo($message)
                        ->setQueueCounts($queueCounts)
                        ->setRecapNotificationRange($recap_notification_range)
                        ->send();
                } catch (\Illuminate\Contracts\Container\BindingResolutionException $th) {
                    $this->console_write()->error(__METHOD__.' | Notification ["'.$notification.'"] not implemented!');
                }
            }
        }
    }
}
