<?php

use Carbon\Carbon;

describe('JobsMonitor Package Test', function () {

    beforeEach(function () {
        setupTestConfig();
        initDbConfig();
        jobsQueueMonitor()->initJobsMonitorStatsDB(true);
        DB::connection('txd_jobs_sqlite')->beginTransaction();
    });

    afterEach(function () {
        DB::connection('txd_jobs_sqlite')->rollBack();
    });

    it('ensures DB is created successfully', function () {

        /** @var \PHPUnit\Framework\TestCase $this */
        jobsQueueMonitor()->initJobsMonitorStatsDB(true);
        $this->assertTrue(file_exists(config('txd_jobs_monitor.sqlite_db_path')));
    });

    it('ensures data are written into DB', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        app()->bind('test_jobs_queue', function () {
            return collect([
                (object) ['queue' => 'default', 'jobs_count' => 6],
                (object) ['queue' => 'emails', 'jobs_count' => 3],
            ]);
        });

        jobsQueueMonitor()->storeCurrentStatus();

        $this->assertTrue(DB::connection('txd_jobs_sqlite')->table('queue_job_status')->exists());

    });

    it('retrieves the average job counts for each queue in the past specified minutes', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        $timestamp = Carbon::now();

        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => $timestamp->copy()->subMinutes(15)],
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => $timestamp->copy()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 10, 'created_at' => $timestamp->copy()->subMinutes(3)],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => $timestamp->copy()->subMinutes(2)],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => $timestamp->copy()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->queueCounts(10);

        expect($result['default'])->toBe(7.5);
        expect($result['emails'])->toBe(20.0);
    });

    it('stores current status in SQLite DB', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        app()->instance('test_jobs_queue', collect([
            (object) ['queue' => 'default', 'jobs_count' => 15],
            (object) ['queue' => 'emails', 'jobs_count' => 25],
        ]));

        jobsQueueMonitor()->storeCurrentStatus();

        $result = DB::connection('txd_jobs_sqlite')->table('queue_job_status')->get();

        expect($result)->toHaveCount(2);
        expect($result->pluck('job_count')->toArray())->toContain(15, 25);
    });

    it('retrieves the number of jobs per queue', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(15)],
            ['queue_name' => 'emails', 'job_count' => 2, 'created_at' => Carbon::now()],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => Carbon::now()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->jobsPerQueue(10);
        expect(count($result['default']))->toBe(2);
        expect(count($result['emails']))->toBe(1);
    });

    it('retrieves the total number of jobs', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(15)],
            ['queue_name' => 'emails', 'job_count' => 2, 'created_at' => Carbon::now()],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => Carbon::now()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->totalJobs(10);
        expect($result)->toBe(6);
    });

    it('test trigger per queue', function ($queueCounts, $maxCountPerQueue, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $result = jobsQueueMonitor()->evalQueueTrigger($queueCounts, $maxCountPerQueue);

        expect($result['status'])->toBe($expected);
    })->with([
        [
            [
                'default' => 6,
                'emails' => 3,
            ], 5, true,
        ],
        [
            [
                'default' => 3,
                'emails' => 3,
            ], 5, false,
        ],
    ]);

    it('test total jobs trigger', function ($totalJobs, $maxTotalCount, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $result = jobsQueueMonitor()->evalTotalJobs($totalJobs, $maxTotalCount);

        expect($result['status'])->toBe($expected);
    })->with([
        [6, 5, true],
        [3, 5, false],
    ]);

    it('test trigger up and down', function ($notify_alert_up, $last_notify_up, $last_notify_down, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('notifications')->insert([
            ['action' => 'notify_up', 'executed' => $last_notify_up],
            ['action' => 'notify_down', 'executed' => $last_notify_down],
        ]);

        $result = jobsQueueMonitor()->evalAndSetTriggers($notify_alert_up);

        expect($result['trigger_up'])->toBe($expected['trigger_up']);
        expect($result['trigger_down'])->toBe($expected['trigger_down']);
    })->with([
        [true, false, false, ['trigger_up' => true, 'trigger_down' => false]],
        [false, false, false, ['trigger_up' => false, 'trigger_down' => true]],
        [true, true, false, ['trigger_up' => false, 'trigger_down' => false]],
        [false, false, true, ['trigger_up' => false, 'trigger_down' => false]],
    ]);

});
