<?php

namespace Txd\JobsMonitor\Classes;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;

class JobsQueueMonitor
{
    /**
     * Retrieves the average job counts for each queue in the past specified minutes.
     *
     * This method connects to the 'txd_jobs_sqlite' database and queries the 'queue_job_status' table
     * to calculate the average job count for each queue within the given time frame.
     *
     * @param  int  $minutes  The number of minutes to look back from the current time.
     * @return array An associative array where the keys are queue names and the values are the average job counts.
     */
    public function queueCounts(int $minutes): array
    {
        return DB::connection('txd_jobs_sqlite')->table('queue_job_status')
            ->select('queue_name', DB::connection('txd_jobs_sqlite')->raw('avg(job_count) as job_count'))
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            ->groupBy('queue_name')
            ->get()
            ->keyBy('queue_name')
            ->map(function ($queue_info) {
                return $queue_info->job_count;
            })
            ->toArray();
    }

    /**
     * Retrieves the number of jobs per queue created within the specified time frame.
     *
     * This method queries the 'queue_job_status' table in the 'txd_jobs_sqlite' database
     * to fetch jobs that were created within the last specified number of minutes.
     * The results are grouped by the queue name.
     *
     * @param  int  $minutes  The number of minutes to look back from the current time.
     * @return array An associative array where the keys are queue names and the values are arrays of jobs.
     */
    public function jobsPerQueue(int $minutes): array
    {
        return DB::connection('txd_jobs_sqlite')->table('queue_job_status')
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            ->get()
            ->groupBy('queue_name')
            ->toArray();
    }

    /**
     * Retrieves the average number of jobs per queue in the last specified minutes.
     *
     * This method connects to the 'txd_jobs_sqlite' database and queries the 'queue_job_status' table
     * to calculate the average number of jobs for each queue within the given time frame.
     *
     * @param  int  $minutes  The number of minutes to look back from the current time.
     * @return int The total number of jobs.
     */
    public function totalJobs(int $minutes): int
    {
        return array_sum(DB::connection('txd_jobs_sqlite')->table('queue_job_status')
            ->select('queue_name', DB::connection('txd_jobs_sqlite')->raw('avg(job_count) as total_jobs'))
            ->where('created_at', '>=', Carbon::now()->subMinutes($minutes))
            ->groupBy('queue_name')
            ->get()
            ->map(function ($queue_info) {
                return $queue_info->total_jobs;
            })
            ->toArray());
    }

    public function evalQueueTrigger(array $queueCounts, int $maxCountPerQueue): array
    {
        $result = false;
        $notification_messages = [];

        foreach ($queueCounts as $queue => $avgCount) {
            $avgCount = round((float) $avgCount, 2);
            if ($avgCount > (float) $maxCountPerQueue) {
                $notification_messages[] = "Queue {$queue} exceeded its limit of ".$maxCountPerQueue.' jobs [average: '.number_format($avgCount, 2).']';
                $result = true;
            } else {
                $notification_messages[] = "Queue {$queue} is under its trigger value of {$maxCountPerQueue} jobs [average: ".number_format($avgCount, 2).']';
            }
        }

        return ['status' => $result, 'messages' => $notification_messages];
    }

    public function evalStaleJobsQueues(int $minutes): array
    {
        // otteniamo un array con il nome della coda e il numero di job per le varie occorrenze registrate
        $jobsCount = collect($this->jobsPerQueue($minutes))->mapWithKeys(function ($job, $key) {
            return [$key => collect($job)->map(function ($item) {
                return $item->job_count;
            })->toArray()];
        });

        $result = $jobsCount->filter(function ($n_jobs) {
            return count($n_jobs) > 1 && count(array_unique($n_jobs)) === 1;
        });

        return ['status' => $result->isNotEmpty(), 'stale_queues' => $result];
    }

    public function evalNotifications(?bool $status_jobs = null, ?bool $status_queue = null, ?bool $status_stale = null): array
    {
        $notify_alert_up = false;
        $notify_alert_stale = false;

        if ($status_jobs) {
            $notify_alert_up = true;
        }

        if ($status_queue) {
            $notify_alert_up = true;
        }

        if ($status_stale) {
            $notify_alert_stale = true;
        }

        $notify_alert_up = $notify_alert_up && ! $notify_alert_stale; //se ci sono entrambi, prevale il notify_alert_stale

        return ['notify_alert_up' => $notify_alert_up, 'notify_alert_stale' => $notify_alert_stale];
    }

    public function evalTotalJobs(int $totalJobs, int $maxTotalCount): array
    {
        $result = false;
        $notification_messages = [];

        if ($totalJobs > $maxTotalCount) {
            $notification_messages[] = 'Total jobs count [average '.number_format($totalJobs, 2).'] exceeded the limit of ['.number_format($maxTotalCount, 0).'] jobs';
            $result = true;
        } else {
            $notification_messages[] = 'Total jobs count [average '.number_format($totalJobs, 2).'] is under the trigger value of ['.number_format($maxTotalCount, 0).'] jobs';
        }

        return ['status' => $result, 'messages' => $notification_messages];
    }

    public function evalAndSetTriggers(array $triggers): array
    {
        [
            'notify_alert_up' => $notify_alert_up,
            'notify_alert_stale' => $notify_alert_stale,
        ] = $triggers;

        $trigger_up = false;
        $trigger_down = false;
        $trigger_stale = false;

        $last_notify_up = optional(DB::connection('txd_jobs_sqlite')->table('notifications')->where('action', 'notify_up')->first())->executed ?? false;
        $last_notify_down = optional(DB::connection('txd_jobs_sqlite')->table('notifications')->where('action', 'notify_down')->first())->executed ?? false;
        $last_notify_stale = optional(DB::connection('txd_jobs_sqlite')->table('notifications')->where('action', 'notify_stale')->first())->executed ?? false;

        if ($notify_alert_up && ! $last_notify_up) {
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_up'], ['executed' => true]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_down'], ['executed' => false]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_stale'], ['executed' => false]);
            $trigger_up = true;
        }

        if (! $notify_alert_up && ! $last_notify_down) {
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_up'], ['executed' => false]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_down'], ['executed' => true]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_stale'], ['executed' => false]);
            $trigger_down = true;
        }

        if ($notify_alert_stale && ! $last_notify_stale) {
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_up'], ['executed' => false]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_down'], ['executed' => false]);
            DB::connection('txd_jobs_sqlite')->table('notifications')->updateOrInsert(['action' => 'notify_stale'], ['executed' => true]);
            $trigger_stale = true;
        }

        return ['trigger_up' => $trigger_up, 'trigger_down' => $trigger_down, 'trigger_stale' => $trigger_stale];
    }

    /**
     * Get the current status of the job queues.
     *
     * This method retrieves the number of jobs in each queue based on the configured queue driver.
     * It supports 'test', 'database', and 'redis' queue drivers.
     *
     * @return array An associative array where the keys are the queue names and the values are the job counts.
     *
     * @throws Exception If the configured queue driver is not implemented.
     */
    public function getCurrentQueueStatus(): array
    {
        switch (config('queue.default')) {
            case 'test':
                //for testing purposes only: before the test, bind the test_jobs_queue to a collection of jobs
                $jobs_per_queue = app('test_jobs_queue')
                    ->keyBy('queue')
                    ->map(function ($queue_info) {
                        return $queue_info->jobs_count;
                    })
                    ->toArray();
                break;
            case 'database':
                $jobs_per_queue = DB::table('jobs')
                    ->select('queue', DB::raw('count(*) as jobs_count'))
                    ->groupBy('queue')
                    ->get()
                    ->keyBy('queue')
                    ->map(function ($queue_info) {
                        return $queue_info->jobs_count;
                    })
                    ->toArray();
                break;

            case 'redis':
                $queues = $this->getHorizonQueues();

                $jobs_per_queue = collect($queues)->mapWithKeys(function ($queue) {
                    $jobsCount = Redis::connection('default')->llen('queues:'.$queue);

                    return [$queue => $jobsCount];
                })->toArray();
                break;
            default:
                throw new Exception('Queue driver not implemented: '.config('queue.default'));
                break;
        }

        return $jobs_per_queue;
    }

    /**
     * Store current status in SQLite DB.
     */
    public function storeCurrentStatus()
    {
        $timestamp = Carbon::now();

        $jobs_per_queue = $this->getCurrentQueueStatus();

        foreach ($jobs_per_queue as $queue => $count) {
            DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
                'queue_name' => $queue,
                'job_count' => $count,
                'created_at' => $timestamp,
            ]);
        }
    }

    public function getHorizonQueues(): array
    {
        return collect(config('horizon.defaults', []))->map(function ($queue) {
            return $queue['queue'] ?? [];
        })->flatten()->toArray();
    }

    /**
     * Initialize the SQLite database if not already created.
     */
    public function initJobsMonitorStatsDB($fresh = false)
    {

        if ($fresh) {
            if (file_exists(config('txd_jobs_monitor.sqlite_db_path'))) {
                unlink(config('txd_jobs_monitor.sqlite_db_path'));
            }
        }

        if (! file_exists(config('txd_jobs_monitor.sqlite_db_path'))) {
            touch(config('txd_jobs_monitor.sqlite_db_path'));
        }

        if (! DB::connection('txd_jobs_sqlite')->getSchemaBuilder()->hasTable('queue_job_status')) {
            DB::connection('txd_jobs_sqlite')->getSchemaBuilder()->create('queue_job_status', function ($table) {
                $table->unsignedInteger('id')->autoIncrement();
                $table->string('queue_name');
                $table->integer('job_count');
                $table->timestamp('created_at');
            });
        }

        if (! DB::connection('txd_jobs_sqlite')->getSchemaBuilder()->hasTable('notifications')) {
            DB::connection('txd_jobs_sqlite')->getSchemaBuilder()->create('notifications', function ($table) {
                $table->string('action')->primary();
                $table->boolean('executed');
            });
        }
    }
}
