<?php

use Carbon\Carbon;
use Mockery\MockInterface;
use Tests\TestCase;
use Txd\JobsMonitor\Commands\JobsQueueMonitorCommand;
use Txd\JobsMonitor\Notifications\Email;

describe('JobsMonitor Package Test', function () {

    beforeEach(function () {
        setupTestConfig();
        initDbConfig();
        jobsQueueMonitor()->initJobsMonitorStatsDB(true);
        DB::connection('txd_jobs_sqlite')->beginTransaction();
    });

    afterEach(function () {
        DB::connection('txd_jobs_sqlite')->rollBack();
    });

    it('ensures DB is created successfully', function () {

        /** @var \PHPUnit\Framework\TestCase $this */
        jobsQueueMonitor()->initJobsMonitorStatsDB(true);
        $this->assertTrue(file_exists(config('txd_jobs_monitor.sqlite_db_path')));
    });

    it('ensures data are written into DB', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        app()->bind('test_jobs_queue', function () {
            return collect([
                (object) ['queue' => 'default', 'jobs_count' => 6],
                (object) ['queue' => 'emails', 'jobs_count' => 3],
            ]);
        });

        jobsQueueMonitor()->storeCurrentStatus();

        $this->assertTrue(DB::connection('txd_jobs_sqlite')->table('queue_job_status')->exists());

    });

    it('retrieves the average job counts for each queue in the past specified minutes', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        $timestamp = Carbon::now();

        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => $timestamp->copy()->subMinutes(15)],
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => $timestamp->copy()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 10, 'created_at' => $timestamp->copy()->subMinutes(3)],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => $timestamp->copy()->subMinutes(2)],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => $timestamp->copy()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->queueCounts(10);

        expect($result['default'])->toBe(7.5);
        expect($result['emails'])->toBe(20.0);
    });

    it('stores current status in SQLite DB', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        app()->instance('test_jobs_queue', collect([
            (object) ['queue' => 'default', 'jobs_count' => 15],
            (object) ['queue' => 'emails', 'jobs_count' => 25],
        ]));

        jobsQueueMonitor()->storeCurrentStatus();

        $result = DB::connection('txd_jobs_sqlite')->table('queue_job_status')->get();

        expect($result)->toHaveCount(2);
        expect($result->pluck('job_count')->toArray())->toContain(15, 25);
    });

    it('retrieves the number of jobs per queue', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(15)],
            ['queue_name' => 'emails', 'job_count' => 2, 'created_at' => Carbon::now()],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => Carbon::now()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->jobsPerQueue(10);
        expect(count($result['default']))->toBe(2);
        expect(count($result['emails']))->toBe(1);
    });

    it('retrieves the total number of jobs', function () {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
            ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(5)],
            ['queue_name' => 'default', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(15)],
            ['queue_name' => 'emails', 'job_count' => 2, 'created_at' => Carbon::now()],
            ['queue_name' => 'emails', 'job_count' => 20, 'created_at' => Carbon::now()->subMinutes(20)],
        ]);

        $result = jobsQueueMonitor()->totalJobs(10);
        expect($result)->toBe(6);
    });

    it('test trigger per queue', function ($queueCounts, $maxCountPerQueue, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $result = jobsQueueMonitor()->evalQueueTrigger($queueCounts, $maxCountPerQueue);
        $notifications = jobsQueueMonitor()->evalNotifications(null, $result['status'], null);

        expect($result['status'])->toBe($expected);
        expect($notifications['notify_alert_up'])->toBe($expected);
    })->with([
        [
            [
                'default' => 6,
                'emails' => 3,
            ], 5, true,
        ],
        [
            [
                'default' => 3,
                'emails' => 3,
            ], 5, false,
        ],
    ]);

    it('test total jobs trigger', function ($totalJobs, $maxTotalCount, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $result = jobsQueueMonitor()->evalTotalJobs($totalJobs, $maxTotalCount);
        $notifications = jobsQueueMonitor()->evalNotifications($result['status'], null, null);

        expect($result['status'])->toBe($expected);
        expect($notifications['notify_alert_up'])->toBe($expected);
    })->with([
        [6, 5, true],
        [3, 5, false],
    ]);

    it('test full notifications status', function ($totalJobs, $maxTotalJobs, $queueCounts, $maxCountPerQueue, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $result = jobsQueueMonitor()->evalNotifications(
            jobsQueueMonitor()->evalTotalJobs($totalJobs, $maxTotalJobs)['status'],
            jobsQueueMonitor()->evalQueueTrigger($queueCounts, $maxCountPerQueue)['status'],
            jobsQueueMonitor()->evalStaleJobsQueues(5)['status']
        );

        expect($result['notify_alert_up'])->toBe($expected['notify_alert_up']);
        expect($result['notify_alert_stale'])->toBe($expected['notify_alert_stale']);
    })->with([
        [6, 5, ['default' => 6, 'emails' => 3], 5, ['notify_alert_up' => true, 'notify_alert_stale' => false]],
        [3, 5, ['default' => 3, 'emails' => 3], 5, ['notify_alert_up' => false, 'notify_alert_stale' => false]],
        [6, 5, ['default' => 3, 'emails' => 3], 5, ['notify_alert_up' => true, 'notify_alert_stale' => false]],
        [3, 5, ['default' => 6, 'emails' => 3], 5, ['notify_alert_up' => true, 'notify_alert_stale' => false]],
        [3, 5, ['default' => 3, 'emails' => 6], 5, ['notify_alert_up' => true, 'notify_alert_stale' => false]],
        [3, 5, ['default' => 3, 'emails' => 3], 5, ['notify_alert_up' => false, 'notify_alert_stale' => false]],
    ]);

    it('test evalStaleJobs - single queue', function ($minutes, $counts, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $date = Carbon::now();
        foreach (collect($counts)->reverse() as $n) {
            DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
                ['queue_name' => 'default', 'job_count' => $n, 'created_at' => $date->subMinute()],
            ]);
        }

        $result = jobsQueueMonitor()->evalStaleJobsQueues($minutes)['status'];

        expect($result)->toBe($expected);
    })->with([
        [15, [2, 2, 2, 2], true],
        [15, [2, 2, 3, 1], false],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 1], true],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 2], false],
    ]);

    it('test evalStaleJobs - multi queue', function ($minutes, $counts_queue1, $counts_queue2, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        $date = Carbon::now();
        foreach (collect($counts_queue1)->reverse() as $n) {
            DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
                ['queue_name' => 'default', 'job_count' => $n, 'created_at' => $date->subMinute()],
            ]);
        }

        foreach (collect($counts_queue2)->reverse() as $n) {
            DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert([
                ['queue_name' => 'emails', 'job_count' => $n, 'created_at' => $date->subMinute()],
            ]);
        }

        $result = jobsQueueMonitor()->evalStaleJobsQueues($minutes)['status'];

        expect($result)->toBe($expected);
    })->with([
        [15, [2, 2, 2, 2], [2, 2, 2, 2], true],
        [15, [2, 2, 3, 1], [2, 2, 2, 2], true],
        [15, [4, 4, 4, 4], [2, 2, 2, 1], true],
        [15, [2, 2, 3, 1], [2, 2, 3, 1], false],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 1], [2, 2, 2, 2, 2, 2, 2, 2, 2], true],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 2], [2, 2, 2, 2, 2, 2, 2, 2, 2], false],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 1], [2, 2, 2, 2, 2, 2, 2, 2, 2], true],
        [5, [2, 2, 3, 1, 1, 1, 1, 1, 2], [2, 2, 2, 2, 2, 2, 2, 2, 2], false],
    ]);

    it('test evalAndSetTriggers', function ($notifications, $last_notify_up, $last_notify_down, $last_notify_stale, $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('notifications')->insert([
            ['action' => 'notify_up', 'executed' => $last_notify_up],
            ['action' => 'notify_down', 'executed' => $last_notify_down],
            ['action' => 'notify_stale', 'executed' => $last_notify_stale],
        ]);

        $result = jobsQueueMonitor()->evalAndSetTriggers($notifications);

        expect($result['trigger_up'])->toBe($expected['trigger_up']);
        expect($result['trigger_down'])->toBe($expected['trigger_down']);
        expect($result['trigger_stale'])->toBe($expected['trigger_stale']);
    })->with([
        [['notify_alert_up' => true, 'notify_alert_stale' => false], false, false, false, ['trigger_up' => true, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, false, false, ['trigger_up' => false, 'trigger_down' => true, 'trigger_stale' => false]],
        [['notify_alert_up' => true, 'notify_alert_stale' => false], true, false, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, true, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => true], false, true, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => true]],
        [['notify_alert_up' => false, 'notify_alert_stale' => true], false, true, true, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, true, true, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
    ]);

    it('test trigger up and down', function (array $notifications, bool $last_notify_up, bool $last_notify_down, bool $last_notify_stale, array $expected) {
        /** @var \PHPUnit\Framework\TestCase $this */
        DB::connection('txd_jobs_sqlite')->table('notifications')->insert([
            ['action' => 'notify_up', 'executed' => $last_notify_up],
            ['action' => 'notify_down', 'executed' => $last_notify_down],
            ['action' => 'notify_stale', 'executed' => $last_notify_stale],
        ]);

        $result = jobsQueueMonitor()->evalAndSetTriggers($notifications);

        expect($result['trigger_up'])->toBe($expected['trigger_up']);
        expect($result['trigger_down'])->toBe($expected['trigger_down']);
        expect($result['trigger_stale'])->toBe($expected['trigger_stale']);
    })->with([
        [['notify_alert_up' => true, 'notify_alert_stale' => false], false, false, false, ['trigger_up' => true, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, false, false, ['trigger_up' => false, 'trigger_down' => true, 'trigger_stale' => false]],
        [['notify_alert_up' => true, 'notify_alert_stale' => false], true, false, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, true, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => true], false, true, false, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => true]],
        [['notify_alert_up' => false, 'notify_alert_stale' => true], false, true, true, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
        [['notify_alert_up' => false, 'notify_alert_stale' => false], false, true, true, ['trigger_up' => false, 'trigger_down' => false, 'trigger_stale' => false]],
    ]);

    it('test full command execution txd:jobs-monitor', function ($queue_jobs_status, $test_jobs_queue, $expected) {
        /** @var TestCase $this */
        app()->bind('test_jobs_queue', function () use ($test_jobs_queue) {
            return $test_jobs_queue;
        });

        foreach ($queue_jobs_status as $queue_job_status) {
            DB::connection('txd_jobs_sqlite')->table('queue_job_status')->insert($queue_job_status);
        }

        $trigger_level = null;
        $this->mock(Email::class, function (MockInterface $mock) use (&$trigger_level) {
            $mock->shouldReceive(
                'setInfo',
                'setQueueCounts',
                'setTotalJobs',
                'setRecapNotificationRange',
                'setStaleQueues',
            )->andReturnSelf();

            $mock->shouldReceive(
                'setNotificationLevel',
            )
                ->withArgs(function ($value_received_by_method) use (&$trigger_level) {
                    // Set the notificationLevel based on the argument received
                    $trigger_level = $value_received_by_method;

                    return true; // Returning true means the method will accept any argument
                })
                ->andReturnSelf();

            $mock->shouldReceive('send');
        });

        app(JobsQueueMonitorCommand::class)->handle();

        expect($trigger_level)->toBe($expected);

    })
        ->with([
            [
                [
                    ['queue_name' => 'default', 'job_count' => 15, 'created_at' => Carbon::now()->subMinutes(4)],
                    ['queue_name' => 'default', 'job_count' => 15, 'created_at' => Carbon::now()->subMinutes(2)],
                    ['queue_name' => 'emails', 'job_count' => 30, 'created_at' => Carbon::now()->subMinutes(4)],
                    ['queue_name' => 'emails', 'job_count' => 30, 'created_at' => Carbon::now()->subMinutes(2)],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 6],
                    (object) ['queue' => 'emails', 'jobs_count' => 3],
                ]),
                'TRIGGER_UP',
            ],
            [
                [
                    ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()->subMinutes(4)],
                    ['queue_name' => 'default', 'job_count' => 5, 'created_at' => Carbon::now()],
                    ['queue_name' => 'emails', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(4)],
                    ['queue_name' => 'emails', 'job_count' => 3, 'created_at' => Carbon::now()],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 5],
                    (object) ['queue' => 'emails', 'jobs_count' => 3],
                ]),
                'TRIGGER_STALE',
            ],
            [
                [
                    ['queue_name' => 'default', 'job_count' => 10, 'created_at' => Carbon::now()->subMinutes(10)],
                    ['queue_name' => 'default', 'job_count' => 10, 'created_at' => Carbon::now()],
                    ['queue_name' => 'emails', 'job_count' => 5, 'created_at' => Carbon::now()->subMinutes(10)],
                    ['queue_name' => 'emails', 'job_count' => 5, 'created_at' => Carbon::now()],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 12],
                    (object) ['queue' => 'emails', 'jobs_count' => 6],
                ]),
                'TRIGGER_UP',
            ],
            [
                [
                    ['queue_name' => 'default', 'job_count' => 2, 'created_at' => Carbon::now()->subMinutes(2)],
                    ['queue_name' => 'default', 'job_count' => 2, 'created_at' => Carbon::now()],
                    ['queue_name' => 'emails', 'job_count' => 1, 'created_at' => Carbon::now()->subMinutes(2)],
                    ['queue_name' => 'emails', 'job_count' => 1, 'created_at' => Carbon::now()],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 3],
                    (object) ['queue' => 'emails', 'jobs_count' => 2],
                ]),
                'TRIGGER_DOWN',
            ],
            [
                [
                    ['queue_name' => 'default', 'job_count' => 8, 'created_at' => Carbon::now()->subMinutes(8)],
                    ['queue_name' => 'default', 'job_count' => 8, 'created_at' => Carbon::now()],
                    ['queue_name' => 'emails', 'job_count' => 4, 'created_at' => Carbon::now()->subMinutes(8)],
                    ['queue_name' => 'emails', 'job_count' => 4, 'created_at' => Carbon::now()],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 9],
                    (object) ['queue' => 'emails', 'jobs_count' => 5],
                ]),
                'TRIGGER_UP',
            ],
            [
                [
                    ['queue_name' => 'default', 'job_count' => 7, 'created_at' => Carbon::now()->subMinutes(7)],
                    ['queue_name' => 'default', 'job_count' => 7, 'created_at' => Carbon::now()],
                    ['queue_name' => 'emails', 'job_count' => 3, 'created_at' => Carbon::now()->subMinutes(7)],
                    ['queue_name' => 'emails', 'job_count' => 3, 'created_at' => Carbon::now()],
                ],
                collect([
                    (object) ['queue' => 'default', 'jobs_count' => 8],
                    (object) ['queue' => 'emails', 'jobs_count' => 4],
                ]),
                'TRIGGER_UP',
            ],
        ]);

});
