<?php

namespace Txd\JobsMonitor\Notifications;

use Carbon\Carbon;
use Exception;
use Mail;
use Txd\JobsMonitor\Contracts\NotificationContract;

class Email implements NotificationContract
{
    protected $config;

    protected $notificationLevel;

    protected $info;

    protected $queueCounts;

    protected $totalJobs;

    protected $recapNotificationRange;

    protected $staleQueues;

    public function __construct()
    {
        $this->config = config('txd_jobs_monitor.notifications.email');
    }

    public function setNotificationLevel(string $notificationLevel): NotificationContract
    {
        $this->notificationLevel = $notificationLevel;

        return $this;
    }

    public function setInfo(string $info): NotificationContract
    {
        $this->info = $info;

        return $this;
    }

    public function setQueueCounts(array $queueCounts): NotificationContract
    {
        $this->queueCounts = $queueCounts;

        return $this;
    }

    public function setStaleQueues($staleQueues): NotificationContract
    {
        $this->staleQueues = $staleQueues;

        return $this;
    }

    public function setTotalJobs(int $totalJobs): NotificationContract
    {
        $this->totalJobs = $totalJobs;

        return $this;
    }

    public function setRecapNotificationRange(int $recapNotificationRange): NotificationContract
    {
        $this->recapNotificationRange = $recapNotificationRange;

        return $this;
    }

    public function send(): bool
    {
        $recipients = $this->getRecipients();

        if (empty($recipients)) {
            throw new Exception(__METHOD__.' | No valid recipients found');
        }

        switch ($this->notificationLevel) {
            case 'TRIGGER_DOWN':
                $testo = '<p>';
                break;
            case 'TRIGGER_UP':
            case 'TRIGGER_STALE':
                $testo = '<p><span style="color: #cc0000; font-weight: bold;">ALERT!</span><br />';
                break;
            default:
                return false;
        }

        $testo .= 'System: <b>'.config('app.name').'</b></p>';
        $testo .= '<p>'.nl2br($this->info).'</p>';

        //TODO: fix email text for TRIGGER_STALE to include stale queues

        $testo .= '<p>JOBS per queue:</p>';
        $tot_jobs = 0;
        $items_count = 0;
        foreach ($this->queueCounts as $queue => $info) {
            $testo .= '<p><b>'.$queue.'</b>:';
            foreach ($info as $value) {
                $testo .= '<br />'.(new Carbon($value->created_at))->format('Y-m-d H:i').': '.$value->job_count;
                $tot_jobs += $value->job_count;
                $items_count++;
            }
            $testo .= '</p>';
        }

        if ($items_count == 0) {
            $items_count = 1;
        }
        $testo .= '<p><b>AVG Total JOBS for the past '.$this->recapNotificationRange.' minutes:</b> '.number_format(round($tot_jobs / $items_count, 2), 2).'</p>';

        Mail::to($recipients)->send(new \Txd\JobsMonitor\Mail\NotificationMail(
            $this->subjectFromLevel($this->notificationLevel),
            $testo
        ));

        return true;
    }

    protected function subjectFromLevel($level): string
    {
        switch ($level) {
            case 'TRIGGER_DOWN':
                $subject = config('app.name').' | Jobs Anomalies Notification - Under Quota';
                break;
            case 'TRIGGER_UP':
                $subject = config('app.name').' | Jobs Anomalies Notification - Over Quota';
                break;
            case 'TRIGGER_STALE':
                $subject = config('app.name').' | Jobs Anomalies Notification - Stale Queues';
                break;
            default:
                $subject = config('app.name').' | Jobs Anomalies Notification';
                break;
        }

        return $subject;
    }

    protected function getRecipients()
    {
        return collect(explode(',', $this->config['recipients']))
            ->map(function ($value) {
                return trim($value);
            })
            ->filter(function ($value) {
                return filter_var($value, FILTER_VALIDATE_EMAIL);
            })->toArray();
    }
}
