<?php

namespace Txd\JobsMonitor\Commands;

use Illuminate\Console\Command;
use Txd\JobsMonitor\Classes\JobsQueueMonitor;

class JobsQueueStatusCommand extends Command
{
    /** @var JobsQueueMonitor */
    protected $jobsQueueMonitor;

    protected $signature = 'txd:jobs-status {--minutes= : Number of minutes to look back}';

    protected $description = 'Show current jobs queue status';

    public function __construct(JobsQueueMonitor $jobsQueueMonitor)
    {
        parent::__construct();

        $this->jobsQueueMonitor = $jobsQueueMonitor;
    }

    /**
     * @return \Illuminate\Console\View\Components\Factory|self
     */
    protected function console_write()
    {
        if (property_exists($this, 'components') && is_a($this->{'components'}, 'Illuminate\Console\View\Components\Factory')) {
            /** @var \Illuminate\Console\View\Components\Factory $callable_message_component */
            $callable_message_component = $this->{'components'};
        } else {
            /** @var self $callable_message_component */
            $callable_message_component = $this;
        }

        return $callable_message_component;
    }

    public function handle()
    {
        if (! config('txd_jobs_monitor.enabled')) {
            $this->console_write()->info('Jobs monitor is disabled');

            return;
        }

        $this->jobsQueueMonitor->initJobsMonitorStatsDB(false);

        $default_minutes = config('txd_jobs_monitor.time_range_min', 5);
        $minutes = $this->option('minutes') ?? $default_minutes;
        $queueCounts = $this->jobsQueueMonitor->queueCounts($minutes);

        if (config('queue.default') == 'redis') {

            $rows = collect($this->jobsQueueMonitor->getHorizonQueues())
                ->map(function ($queue) use ($queueCounts) {
                    return [$queue, number_format(round($queueCounts[$queue] ?? 0, 2), 2)];
                })
                ->toArray();
        } else {
            $rows = collect($this->jobsQueueMonitor->queueCounts($minutes))
                ->filter(function ($value, $queue) {
                    return strlen($queue) > 0;
                })
                ->map(function ($value, $queue) {
                    return [$queue, number_format(round($value, 2), 2)];
                })
                ->toArray();
        }

        $this->line("\nJobs average queue status for the last $minutes minutes:");
        $this->table(
            ['Queue Name', 'AVG Jobs Count'],
            $rows
        );

        $this->line("\nCurrent jobs queue status:");
        $this->table(
            ['Queue Name', '# Jobs Count'],
            collect($this->jobsQueueMonitor->getCurrentQueueStatus())
                ->map(function ($value, $queue) {
                    return [$queue, $value];
                })
                ->toArray()
        );

    }
}
