<?php

namespace Txd\Opcache\Commands;

use GuzzleHttp\Client;
use Illuminate\Console\Command;

class OpcacheStatusCommand extends Command
{
    protected $signature = 'opcache:status';

    protected $description = 'Show OPCache status via HTTP request.';

    public function handle()
    {
        if (property_exists($this, 'components') && is_a($this->{'components'}, 'Illuminate\Console\View\Components\Factory')) {
            /** @var \Illuminate\Console\View\Components\Factory $callable_message_component */
            $callable_message_component = $this->{'components'};
        } else {
            /** @var self $callable_message_component */
            $callable_message_component = $this;
        }

        $client = new Client;
        try {
            $response = $client->request('GET', 'http://127.0.0.1/opcache_status');
            $result = optional(json_decode((string) $response->getBody()))->result ?? null;
            if ($result) {
                $this->displayTables($result);
            } else {
                $callable_message_component->error('OPcache not configured');

                return 2;
            }
        } catch (\Exception $e) {
            $callable_message_component->error($e->getMessage());
        }
    }

    /**
     * Display info tables.
     */
    protected function displayTables($data)
    {
        $general = json_decode(json_encode($data), true);

        foreach (['memory_usage', 'interned_strings_usage', 'opcache_statistics', 'preload_statistics', 'jit'] as $unset) {
            unset($general[$unset]);
        }

        $this->table([], $this->parseTable($general));

        $this->line(PHP_EOL.'Memory usage:');
        $this->table([], $this->parseTable($data->memory_usage));

        if (isset($data->opcache_statistics)) {
            $this->line(PHP_EOL.'Statistics:');
            $this->table([], $this->parseTable($data->opcache_statistics));
        }

        if (isset($data->interned_strings_usage)) {
            $this->line(PHP_EOL.'Interned strings usage:');
            $this->table([], $this->parseTable($data->interned_strings_usage));
        }

        if (isset($data->preload_statistics)) {
            $this->line(PHP_EOL.'Preload statistics:');
            $this->table([], $this->parseTable($data->preload_statistics));
        }
    }

    /**
     * Make up the table for console display.
     *
     *
     * @return array
     */
    protected function parseTable($input)
    {
        $input = (array) $input;
        $bytes = ['used_memory', 'free_memory', 'wasted_memory', 'buffer_size'];
        $times = ['start_time', 'last_restart_time'];

        return array_map(function ($key, $value) use ($bytes, $times) {

            if (in_array($key, $bytes)) {
                $value = number_format($value / 1048576, 2).' MB';
            } elseif (in_array($key, $times)) {
                $value = date('Y-m-d H:i:s', $value);
            } elseif (is_bool($value)) {
                $value = $value ? 'true' : 'false';
            }

            return [
                'key' => $key,
                'value' => is_array($value) ? implode(PHP_EOL, $value) : $value,
            ];
        }, array_keys($input), $input);
    }
}
