<?php

namespace Txd\PassportBackchannel\Client\Guards;

use Carbon\Carbon;
use Closure;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Illuminate\Http\Request;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Contracts\Auth\UserProvider;
use Illuminate\Auth\GuardHelpers;
use Illuminate\Support\Facades\Cache as FacadesCache;
use Illuminate\Support\Facades\Log;
use stdClass;

/**
 * estensione della classe TokenGuard per controllare anche la scadenza del token
 */
class BackchannelGuard implements Guard
{
    use GuardHelpers {
        check as trait_check;
    }

    /**
     * The request instance.
     *
     * @var \Illuminate\Http\Request
     */
    protected $request;

    /**
     * The name of the query string item from the request containing the API token.
     *
     * @var string
     */
    protected $inputKey;

    /**
     * The name of the token "column" in persistent storage.
     *
     * @var string
     */
    protected $storageKey;
	
	protected $apiExpirationField;

	/**
     * Create a new authentication guard.
     *
     * @param  \Illuminate\Contracts\Auth\UserProvider  $provider
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    public function __construct(?UserProvider $provider, Request $request)
    {
        $this->request = $request;
        $this->provider = $provider;
        $this->inputKey = 'token';
        $this->storageKey = config("passport-backchannel-client.user_storage_key");
        $this->apiExpirationField = 'api_token_expire';
    }
    
    public function validateKeySource(?string $jwksSource){
        if(is_null($jwksSource)) return false;
        $allowedDomains = config("passport-backchannel-client.allowed_domains");
        
        foreach($allowedDomains as $domain){
                $host = parse_url($domain, PHP_URL_HOST);
                $scheme = parse_url($domain, PHP_URL_SCHEME);
                if(!in_array($scheme,["http","https"])|| empty($host)){
                    continue;
                }
                $baseDomain = "$scheme://$host";
                if(str_starts_with($jwksSource,$baseDomain)){
                    return true;
                }
        }
        return false;
    }

    protected function decodeToken(string $token): stdClass|false {
        if(!is_null($token)){
            
            $kid = $this->request->header("kid");
            $jwksSource = $this->request->header("key");
            if(!$this->validateKeySource($jwksSource)){
                return false;
            }
            
            $jwks = FacadesCache::remember(config("passport-backchannel-client.cache_key").".keys.$jwksSource",60,function() use($jwksSource){
                return json_decode(file_get_contents($jwksSource),true);
            });
            
            
            
            try{
                $decoded = JWT::decode($token,new Key($jwks["keys"][$kid],"RS256"));
                if(is_null($decoded->exp ?? null) && !is_null(config("passport-backchannel-client.token-max-age")) && Carbon::createFromTimestamp($decoded->iat) < now()->subSeconds(config("passport-backchannel-client.token-max-age"))){    
                    return false;
                }
                return $decoded;
            }catch(\Throwable $e){
                Log::info($e->getMessage());
            }
        }
        
        return false;
    }
    
    public function check()
    {
        if(is_null($this->provider)){
            $token = $this->getTokenForRequest();
            if(!is_string($token)){
                return false;
            }
            $decoded = $this->decodeToken($token);
            return $decoded != false;
        }
        return $this->trait_check();
    }
    
    /**
     * Get the currently authenticated user.
     *
     * @return \Illuminate\Contracts\Auth\Authenticatable|null
     */
    public function user()
    {
        // If we've already retrieved the user for the current request we can just
        // return it back immediately. We do not want to fetch the user data on
        // every call to this method because that would be tremendously slow.
        if (! is_null($this->user)) {
            return $this->user;
        }

        if(is_null($this->provider)){
            return null;
        }
        $user = null;

        $token = $this->getTokenForRequest();
        
        $decoded = $this->decodeToken($token);
        if($decoded != false){ 
             $user = $this->provider->retrieveByCredentials(
                [$this->storageKey => $decoded->sub]
            );
        }else{
            abort(401);
        }

		
//		dd($user);
        return $this->user = $user;
    }

    /**
     * Get the token for the current request.
     *
     * @return string
     */
    public function getTokenForRequest()
    {
        $token = $this->request->query($this->inputKey);

        if (empty($token)) {
            $token = $this->request->input($this->inputKey);
        }

        if (empty($token)) {
            $token = $this->request->bearerToken();
        }

        return $token;
    }

    /**
     * Validate a user's credentials.
     *
     * @param  array  $credentials
     * @return bool
     */
    public function validate(array $credentials = [])
    {
        if (empty($credentials[$this->inputKey])) {
            return false;
        }

        $credentials = [$this->storageKey => $credentials[$this->inputKey]];

		$user = $this->provider->retrieveByCredentials($credentials);
		
		//controlliamo se il token ha una scadenza e nel caso che non sia gia' passata
		if(is_object($user)){
			if($user->{$this->apiExpirationField} !== null && $user->{$this->apiExpirationField} < date("Y-m-d H:i:s")){
				return false;
			}else{
				return true;
			}
		}

        return false;
    }
    

    /**
     * Set the current request instance.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return $this
     */
    public function setRequest(Request $request)
    {
        $this->request = $request;

        return $this;
    }
}
