<?php

namespace Txd\PassportBackchannel;

use App\Models\User;
use DomainException;
use Http;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Database\Eloquent\InvalidCastException;
use Illuminate\Database\Eloquent\MissingAttributeException;
use Illuminate\Support\Facades\Http as FacadesHttp;
use Laravel\Passport\Client;
use LogicException;

use function Illuminate\Filesystem\join_paths;

class BackchannelClient
{
    /** @var null|\Illuminate\Http\Client\PendingRequest */
    protected $_client = null;

    protected ?TokenBuilder $_tokenBuilder = null;

    public function __construct(protected Client $oauthClient) {}

    protected function tokenBuilder($force = false)
    {
        if (is_null($this->_tokenBuilder) || $force) {
            $callback = app("backchannel-identifier-callback");
            $userId = $callback(auth()->user());
            $this->_tokenBuilder = new TokenBuilder($this->oauthClient,$userId);
        }

        return $this->_tokenBuilder;
    }

    protected $_url = null;
    
    

    public function getUrl()
    {
        if (is_null($this->_url)) {
            $confClients = collect(config('services.oauth_clients'))->keyBy('id');
            $this->_url = $confClients[$this->oauthClient->getKey()]['url'] ?? null;
            if (is_null($this->_url)) {
                $url = $this->oauthClient->redirect_uris[0] ?? null;
                $host = parse_url($url, PHP_URL_HOST);
                $scheme = parse_url($url, PHP_URL_SCHEME);
                $this->_url = "$scheme://$host";
            }
        }

        return $this->_url;
    }

    /**
     * client inizializzato con l'autenticazione di backchannel (ma senza forzare il percorso di chiamata).
     *
     * @param  string  $eventName
     * @param  array  $payload
     * @param  null|User  $user
     * @return null|\Illuminate\Http\Client\PendingRequest
     *
     * @throws InvalidCastException
     * @throws MissingAttributeException
     * @throws LogicException
     * @throws DomainException
     */
    protected function client()
    {
        if (is_null($this->_client)) {
            $this->_client = FacadesHttp::withHeaders([
                'kid' => 'oauth-public',
                'key' => route('passport.public_key'),
            ])->withToken($this->tokenBuilder()->token())
                ->baseUrl($this->getUrl());
        }

        return $this->_client;
    }

    public function reset()
    {
        $this->_client = null;
        $this->_tokenBuilder = null;
        $this->oauthClient->refresh();
    }

    /**
     * chiamata alle api /api/backchannel/$eventname. usata per azioni dedicate al canale di backchannel
     *
     * @return mixed
     *
     * @throws InvalidCastException
     * @throws MissingAttributeException
     * @throws LogicException
     * @throws DomainException
     */
    public function backchannelCall(string $eventName, array $payload = [],  $userId = null)
    {
        if (!is_null($userId)) {
            $payload['token'] = $this->tokenBuilder()->token([], $eventName, $userId);
        }

        return $this->client()->post(join_paths('api', 'backchannel', $eventName), $payload);
    }
}
