<?php

namespace Txd;
class txdPdfTools {
    
    protected $api_baseurl;
    protected $develop = true;
    
    protected $local_source_file_path = null;
    protected $source_content = null;
    protected $out_filename = "document.pdf";
    protected $out_force_download = false;
    protected $source_pdf_total_page;
    protected $error_messages = [];
    protected $throw_permission_exception = false;


	public function __construct($throw_permission_exception = null){
        
        // if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
		
		$this->api_baseurl = config("txd_pdf_tools.api_gateway_baseurl");
        $this->develop = config("txd_pdf_tools.develop");
        
        if(is_bool($throw_permission_exception)){
            $this->throw_permission_exception = $throw_permission_exception;
        }else{
            $this->throw_permission_exception = config("txd_pdf_tools.jwt_throw_permission_exception");
        }
	}

    
    /**
     * imposta il contentuto del PDF da modificare
     *
     * @param string $base64_string
     * @return txdPdfTools
     */
    public function set_source_content($base64_string){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->source_content = $base64_string;
        return $this;
    }

    /**
     * imposta il contentuto del PDF partendo da un file locale
     *
     * @param string $local_file_path
     * @return txdPdfTools
     */
    public function set_source_from_local_file($local_file_path){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->local_source_file_path = $local_file_path;
        $this->source_content = file_get_contents($this->local_source_file_path);
        return $this;
    }


    /**
     *
     * @param string $filename
     * @return txdPdfTools
     */
    public function out_filename($filename){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->out_filename = $filename;
        return $this;
    }

    /**
     *
     * @param bool $force_download
     * @return txdPdfTools
     */
    public function out_force_download($force_download){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $this->out_force_download = $force_download;
        return $this;
    }


    public function getErrorMessages(){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }
        return $this->error_messages;
    }


    /**
     * verifica se il PDF di partenza e' gia' adeguato ad essere manipolato dal componente php o se necessita di un "fix" di formato
     * in caso di errore popola l'array error_messages con le info (recuperabile tramite ->getErrorMessages())
     *
     * @param null|string $local_source_file_path percorso del file locale da analizzare (NULL se e' gia' stato impostato con un metodo ad hoc)
     * @return bool
     */
    public function check_pdf($local_source_file_path = null){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if(is_string($local_source_file_path)){
            $this->local_source_file_path = $local_source_file_path;
        }

        if($this->local_source_file_path === null){
            throw new \Exception("Prima di procedere e' necessario impostare il percorso del file da controllare (o passarlo direttamente al metodo)");
        }

        if(!is_file($this->local_source_file_path)){
            throw new \Exception("File non trovato: ".$this->local_source_file_path);
        }

        try {
            $pdf = new \setasign\Fpdi\Fpdi();
            $this->source_pdf_total_page = $pdf->setSourceFile($this->local_source_file_path);
            
            return true;

        } catch (\Exception $th) {
            $this->error_messages[] = $th->getMessage();
            return false;
        }
    }


    /**
     * invoca la funzione per "sistemare" il PDF in modo che sia compatibile con PHP
     *
     * @param boolean $return_as_string [default: FALSE] se impostato a TRUE ritorna il file fixed come stringa, altrimenti come Response
     * @return \Response|string
     */
    public function pdf_fix($return_as_string = false){
        
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->source_content === null){
            throw new \Exception("Prima di procedere e' necessario impostare un file sorgente, ad esempio utilizzando il metodo set_source_content()");
        }

        $endpoint = $this->api_baseurl."/pdfFix".($this->develop ? "-develop" : "");
        $jwt_servizio = "pdfFix".($this->develop ? "-develop" : "");

        return $this->api_call($endpoint, $jwt_servizio, $return_as_string);
    }


    /**
     * invoca la funzione per "sistemare" il PDF in modo che sia compatibile con PHP
     *
     * @param boolean $return_as_string [default: FALSE] se impostato a TRUE ritorna il file fixed come stringa, altrimenti come Response
     * @return \Response|string
     */
    public function pdf_flatten($return_as_string = false){
        
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        if($this->source_content === null){
            throw new \Exception("Prima di procedere e' necessario impostare un file sorgente, ad esempio utilizzando il metodo set_source_content()");
        }

        $endpoint = $this->api_baseurl."/pdfFlatten".($this->develop ? "-develop" : "");
        $jwt_servizio = "pdfFlatten".($this->develop ? "-develop" : "");

        return $this->api_call($endpoint, $jwt_servizio, $return_as_string);
    }



    /**
     * wrapper per invocare le lambda e ottenere il PDF elaborato
     *
     * @param string $endpoint
     * @param string $jwt_servizio
     * @param boolean $return_as_string
     * @return string|\Response
     */
    private function api_call($endpoint, $jwt_servizio, $return_as_string = false){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $client = new \GuzzleHttp\Client();

        $userToken = $this->generate_jwt($jwt_servizio);

        try {
            
            $response = $client->post($endpoint, [
                'headers' => [
                    'Accept' => 'application/json',
                    // 'Content-Type' => 'multipart/form-data',  // <-- commented out this line
                    'Authorization' => 'Bearer '. $userToken,
                ],
                'multipart' => [
                    [
                        'name' => 'uploaded_file',
                        'contents' => $this->source_content,
                    ],
                ],
            ]);

            $headers = [
                "Content-Type"=> "application/pdf",
                "Content-Disposition" => ($this->out_force_download === true ? "attachment; " : "").'filename="'.$this->out_filename.'"',
            ];
            
            if($return_as_string){
                return (string) $response->getBody();
            }else{
                return \Response::make($response->getBody(), 200, $headers);
            }

        } catch(\Exception $e) {
            \Log::error(__METHOD__." | ".$e->getMessage());

            if($this->throw_permission_exception === true){
                throw $e;
            }

            return null;
        }
    }


    /**
     * genera un jwt da utilizzare come Bearer nelle chiamate ai nostri API Gateway
     *
     * @param string $servizio_richiesto indica il servizio per il quale si vuole ottenere il token di accesso (pdfFix | pdfFlatten | ...)
     * @return string jwt token
     */
    private function generate_jwt($servizio_richiesto){
        if(class_exists("jsmDoc")){ \jsmDoc::log_method(__FUNCTION__, get_called_class(), __FILE__); }

        $payload = array(
            "iss" => config("txd_pdf_tools.jwt_api_key"),
            "sub" => $servizio_richiesto,
            "iat" => time(),
            "nbf" => time(),
            "exp" => time() + config("txd_pdf_tools.jwt_time_validity", 300),
        );

        $key = "";

        if(config("txd_pdf_tools.jwt_algo") === "HS256"){

            if(strlen(config("txd_pdf_tools.jwt_secret")) == 0){
                throw new \Exception(__METHOD__." | E' necessario specificare un jwt_secret per utilizzare l'algoritmo HS256");
            }

            $key = config("txd_pdf_tools.jwt_secret");

        }elseif(config("txd_pdf_tools.jwt_algo") === "RS256"){

            if(!is_file(config("txd_pdf_tools.jwt_private_key_path"))){
                throw new \Exception(__METHOD__." | E' necessario specificare un percorso valido per la chiave privata per l'algoritmo RS256");
            }

            $key = file_get_contents(config("txd_pdf_tools.jwt_private_key_path"));

        }else{
            throw new \Exception(__METHOD__." | jwt_algo non supportato: ".config("txd_pdf_tools.jwt_algo"));
        }

        return \Firebase\JWT\JWT::encode($payload, $key, config("txd_pdf_tools.jwt_algo"));
    }
}