<?php

namespace Txd;

use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use Illuminate\Support\Facades\Response;

class txdPdfAcroForm
{
    /** Base URL of the overlay service, e.g. http://pdf_acroform:8080 */
    protected string $baseUrl;

    /** Relative path of the endpoint (default /overlay) */
    protected string $endpoint = '/overlay';

    /** Guzzle client */
    protected ClientInterface $http;

    /** Query options sent to the service */
    protected array $query = [
        // defaults aligned with your Python service
        'placeholder_regex' => '##([A-Za-z0-9_]+)(?::(\d+))?##',
        'default_chars' => 10,
        'font_size' => 10.0,
        'char_width_factor' => 0.6,
        'padding_chars' => 2.0,
        'min_w_mm' => 36.0,
        'min_h_mm' => 4.0,
        'y_offset_up_mm' => 0.5,
    ];

    /** Extra request headers */
    protected array $headers = [
        'Accept' => 'application/pdf',
    ];

    /** Timeout (seconds) */
    protected int $timeout = 180;

    /** If true, rethrow HTTP errors; else return false */
    protected bool $throwOnError = false;

    public function __construct(?ClientInterface $http = null)
    {
        $this->baseUrl = rtrim(config('txd_pdf_tools.pdf_acroform_baseurl'), '/');
        $this->http = $http ?: new Client;
    }

    // -------- Fluent setters (return $this) --------

    public function defaultChars(int $n): self
    {
        $this->query['default_chars'] = max(1, $n);

        return $this;
    }

    public function fontSize(float $pt): self
    {
        $this->query['font_size'] = max(6.0, $pt);

        return $this;
    }

    public function charWidthFactor(float $f): self
    {
        $this->query['char_width_factor'] = max(0.4, min(0.9, $f));

        return $this;
    }

    public function paddingChars(float $p): self
    {
        $this->query['padding_chars'] = max(0.0, $p);

        return $this;
    }

    public function minWidthMm(float $mm): self
    {
        $this->query['min_w_mm'] = max(1.0, $mm);

        return $this;
    }

    public function minHeightMm(float $mm): self
    {
        $this->query['min_h_mm'] = max(1.0, $mm);

        return $this;
    }

    public function yOffsetUpMm(float $mm): self
    {
        $this->query['y_offset_up_mm'] = $mm; // can be negative to move down

        return $this;
    }

    /** Change regex if you ever change placeholder format */
    public function placeholderRegex(string $regex): self
    {
        $this->query['placeholder_regex'] = $regex;

        return $this;
    }

    public function header(string $name, string $value): self
    {
        $this->headers[$name] = $value;

        return $this;
    }

    public function headers(array $headers): self
    {
        $this->headers = array_merge($this->headers, $headers);

        return $this;
    }

    public function timeout(int $seconds): self
    {
        $this->timeout = max(1, $seconds);

        return $this;
    }

    public function throwOnError(bool $throw = true): self
    {
        $this->throwOnError = $throw;

        return $this;
    }

    // -------- Main call --------

    /**
     * @param  string  $sourceFile  path to source PDF OR raw PDF bytes (string) OR base64 of PDF
     * @param  bool  $streamAsPdf  if true returns Laravel Response, else raw bytes
     * @param  string|null  $title  suggested filename in Content-Disposition
     * @return \Illuminate\Http\Response|string|false
     */
    public function generate(string $sourceFile, bool $streamAsPdf = true, ?string $title = null)
    {
        $url = $this->baseUrl.$this->endpoint;
        $qs = http_build_query($this->filterNulls($this->query));
        if ($qs !== '') {
            $url .= '?'.$qs;
        }

        // Decide: filesystem path vs raw bytes (or base64)
        $isPath = $this->looksLikePath($sourceFile) && @is_file($sourceFile) && @is_readable($sourceFile);

        if (!$isPath) {
            // Accept base64 (starts with %PDF -> "JVBERi0")
            if (preg_match('/^\s*JVBERi0/i', $sourceFile) && ($decoded = base64_decode($sourceFile, true)) !== false) {
                $sourceBytes = $decoded;
            } else {
                $sourceBytes = $sourceFile;
            }

            if (!$this->looksLikePdfBytes($sourceBytes)) {
                \Log::error(__METHOD__.': invalid $sourceFile (neither readable file nor PDF bytes/base64)');

                return false;
            }
        }

        $multipart = [
            [
                'name' => 'file', // required by FastAPI endpoint
                'contents' => $isPath ? fopen($sourceFile, 'rb') : $sourceBytes,
                'filename' => $isPath ? basename($sourceFile) : 'document.pdf',
                'headers' => ['Content-Type' => 'application/pdf'],
            ],
        ];

        $options = [
            'headers' => $this->headers,
            'multipart' => $multipart,
            'timeout' => $this->timeout,
            'http_errors' => true,
        ];

        try {
            $res = $this->http->post($url, $options);
            $bytes = (string) $res->getBody();

            if ($streamAsPdf) {
                return \Response::make($bytes, 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline;'.($title ? ' filename="'.$title.'"' : ''),
                ]);
            }

            return $bytes;
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            if ($this->throwOnError) {
                throw $e;
            }
            \Log::error(__METHOD__.': request failed', [
                'exception' => $e,
                'url' => $url,
                'as_path' => $isPath,
            ]);

            return false;
        }
    }

    // -------- Helpers --------

    /**
     * Heuristic: short, path-looking string
     */
    protected function looksLikePath(string $s): bool
    {
        if ($s === '') {
            return false;
        }
        if (strlen($s) > 4096) {
            return false;
        }                     // raw bytes are typically long
        if (strpbrk($s, "\0") !== false) {
            return false;
        }           // null byte -> not a path

        // unix abs/rel, windows drive, or dot-prefix
        return $s[0] === '/' || $s[0] === '.' || $s[0] === '~' ||
            preg_match('/^[A-Za-z]:[\\\\\\/]/', $s) === 1;
    }

    /**
     * Quick check for PDF bytes (allow leading BOM/whitespace). Also works for partial buffers.
     */
    protected function looksLikePdfBytes(string $buf): bool
    {
        if ($buf === '') {
            return false;
        }
        $probe = substr($buf, 0, 1024);
        // trim typical BOM/whitespace
        $probe = ltrim($probe, "\xEF\xBB\xBF\x00\r\n\t \v\f");

        return str_starts_with($probe, '%PDF-');
    }

    protected function filterNulls(array $a): array
    {
        return array_filter($a, static fn ($v) => $v !== null);
    }
}
