<?php

namespace Scribble\Actions;

use Cache;
use Carbon\Carbon;
use Scribble\Enums\Permessi;
use Scribble\Enums\TipiCampi;
use Scribble\Models\Campo;
use Scribble\Models\ScribbleForm;

class GeneraSchemaForm
{
    public function handle(ScribbleForm $scribbleForm, Permessi $permessi, array $campiEsterni = [], array $voci_terminate = [], $noCache = false)
    {
        // if (!config('par.cached_schema')) {
        //     return $this->legacyHandle($scribbleForm, $permessi, $campiEsterni, $voci_terminate);
        // }
        if ($noCache || !config('par.cached_schema')) {
            $cached = $this->localHandle($scribbleForm, $permessi);
        } else {
            $cached = Cache::rememberForever("schema_form.{$scribbleForm->getKey()}.{$permessi->value}", fn () => $this->localHandle($scribbleForm, $permessi));
        }

        foreach ($cached[0] as $pageIdx => $page) {
            foreach ($page['schema'] as $groupIdx => $group) {
                foreach ($group['schema'] as $fieldIdx => $field) {
                    if (!$field['disabled'] && in_array($field['name'], $campiEsterni)) {
                        $cached[0][$pageIdx]['schema'][$groupIdx]['schema'][$fieldIdx]['disabled'] = true;
                    }
                }
            }
        }

        $cached[1] = $cached[1]->mapWithKeys(function ($el, $k) use ($voci_terminate) {
            $el['opzioni'] = $el['opzioni']->filter(fn ($v) => !in_array($v['value'], $voci_terminate));

            return [$k => $el];
        });

        return $cached;
    }

    public function legacyHandle(ScribbleForm $scribbleForm, Permessi $permessi, array $campiEsterni = [], array $voci_terminate = [])
    {

        $campi = $scribbleForm->campi;
        $campi = $campi->keyBy('id');
        $regole = $scribbleForm->regole;

        $campi = $campi->sortBy('ordine')->sortBy('gruppo')->sortBy('pagina');
        $default = $campi->keyBy('nome')->filter(fn ($c) => !is_null($c->default))->map(fn ($c) => $c->default)->toArray();
        $schema = [];

        foreach ($campi->groupBy('pagina') as $pagina => $campiPagina) {
            $pagIndex = (int) $pagina + 1;
            $schema[] = [
                'name' => "Pagina $pagIndex",
                'type' => 'group',
                'class' => 'page-container',
                'schema' => $campiPagina->groupBy('gruppo')->map(function ($val, $key) use ($permessi, $campiEsterni) {
                    return [
                        'type' => 'group',
                        'class' => 'group-container bg-booking',
                        'schema' => $val->keyBy('nome')->map(fn ($campo) => $this->mapCampo($campo, $permessi, $campiEsterni))->toArray(),
                    ];
                })->toArray(),
            ];
        }

        $liste = $campi->filter(fn ($campo) => $campo->tipo === TipiCampi::Select->value)
            ->mapWithKeys(function ($campo) use ($voci_terminate, $schema) {
                $opzioniSelect = $campo->voci->sortBy('ordine')
                    ->filter(fn ($voce) => !in_array($voce->getKey(), $voci_terminate))
                    ->map(fn ($voce) => ['value' => $voce->valore, 'label' => $voce->etichetta])->values();
                if (!in_array('required', $schema[$campo->pagina]['schema'][$campo->gruppo]['schema'][$campo->nome]['rules'] ?? [])) {
                    $opzioniSelect->prepend(['value' => null, 'label' => [
                        'it' => '--',
                        'en' => '--',
                        'es' => '--',
                        'pt' => '--',
                    ]]);
                }

                return [$campo->nome => $opzioniSelect];
            })
            ->map(fn ($opts) => ['opzioni' => $opts, 'dipendenze' => []]);

        foreach ($regole as $regola) {
            $condizione = $this->condizioneRicorsiva($regola->condizione, $campi);
            if ($regola->condizione['type'] == 'or') {
                $condizione = [$condizione];
            }

            foreach ($regola->mostra_campi as $id) {

                $el = $schema[$campi[$id]->pagina]['schema'][$campi[$id]->gruppo]['schema'][$campi[$id]->nome];
                if (is_null($el['conditions'] ?? null)) {
                    $el['conditions'] = [];
                }

                // le condizioni di visibilità sui campi date da regole differenti sono valutate in AND
                $el['conditions'] = [...$el['conditions'], ...$condizione];

                $schema[$campi[$id]->pagina]['schema'][$campi[$id]->gruppo]['schema'][$campi[$id]->nome] = $el;
            }

            foreach ($regola->filtra_voci as $azione) {
                $campo = $campi[$azione['campo']];
                $items = $liste[$campo->nome];
                $items['dipendenze'] = array_values(array_unique([...($items['dipendenze'] ?? []), ...array_map(fn ($id) => $campi[$id]->nome, $regola->get_campi_in_condizione())]));

                foreach ($azione['voci'] as $voce) {
                    $idx = $items['opzioni']->search(fn ($v) => $v['value'] == $voce);
                    if ($idx !== false) {
                        $voce = $items['opzioni'][$idx];
                        $voce['condizioni'] = [...($voce['condizioni'] ?? []), ['condizione' => $condizione, 'comportamento' => $azione['comportamento']]];
                        $items['opzioni'][$idx] = $voce;
                    }
                }
                $liste[$campo->nome] = $items;
            }
        }

        return [$schema, $liste, $default];
    }

    public function localHandle(ScribbleForm $scribbleForm, Permessi $permessi)
    {

        $campi = $scribbleForm->campi;
        $campi = $campi->keyBy('id');
        $regole = $scribbleForm->regole;

        $campi = $campi->sortBy('ordine')->sortBy('gruppo')->sortBy('pagina');
        $default = $campi->keyBy('nome')->filter(fn ($c) => !is_null($c->default))->map(fn ($c) => $c->default)->toArray();
        $schema = [];

        foreach ($campi->groupBy('pagina') as $pagina => $campiPagina) {
            $pagIndex = (int) $pagina + 1;
            $schema[] = [
                'name' => "Pagina $pagIndex",
                'type' => 'group',
                'class' => 'page-container',
                'schema' => $campiPagina->groupBy('gruppo')->map(function ($val, $key) use ($permessi) {
                    return [
                        'type' => 'group',
                        'class' => 'group-container bg-booking',
                        'schema' => $val->keyBy('nome')->map(fn ($campo) => $this->mapCampo($campo, $permessi))->toArray(),
                    ];
                })->toArray(),
            ];
        }

        $liste = $campi->filter(fn ($campo) => $campo->tipo === TipiCampi::Select->value)
            ->mapWithKeys(function ($campo) use ($schema) {
                $opzioniSelect = $campo->voci->sortBy('ordine')
                    ->map(fn ($voce) => ['value' => $voce->valore, 'label' => $voce->etichetta])->values();
                if (!in_array('required', $schema[$campo->pagina]['schema'][$campo->gruppo]['schema'][$campo->nome]['rules'] ?? [])) {
                    $opzioniSelect->prepend(['value' => null, 'label' => [
                        'it' => '--',
                        'en' => '--',
                        'es' => '--',
                        'pt' => '--',
                    ]]);
                }

                return [$campo->nome => $opzioniSelect];
            })
            ->map(fn ($opts) => ['opzioni' => $opts, 'dipendenze' => []]);

        foreach ($regole as $regola) {
            $condizione = $this->condizioneRicorsiva($regola->condizione, $campi);
            if ($regola->condizione['type'] == 'or') {
                $condizione = [$condizione];
            }

            foreach ($regola->mostra_campi as $id) {

                $el = $schema[$campi[$id]->pagina]['schema'][$campi[$id]->gruppo]['schema'][$campi[$id]->nome];
                if (is_null($el['conditions'] ?? null)) {
                    $el['conditions'] = [];
                }

                // le condizioni di visibilità sui campi date da regole differenti sono valutate in AND
                $el['conditions'] = [...$el['conditions'], ...$condizione];

                $schema[$campi[$id]->pagina]['schema'][$campi[$id]->gruppo]['schema'][$campi[$id]->nome] = $el;
            }

            foreach ($regola->filtra_voci as $azione) {
                $campo = $campi[$azione['campo']];
                $items = $liste[$campo->nome];
                $items['dipendenze'] = array_values(array_unique([...($items['dipendenze'] ?? []), ...array_map(fn ($id) => $campi[$id]->nome, $regola->get_campi_in_condizione())]));

                foreach ($azione['voci'] as $voce) {
                    $idx = $items['opzioni']->search(fn ($v) => $v['value'] == $voce);
                    if ($idx !== false) {
                        $voce = $items['opzioni'][$idx];
                        $voce['condizioni'] = [...($voce['condizioni'] ?? []), ['condizione' => $condizione, 'comportamento' => $azione['comportamento']]];
                        $items['opzioni'][$idx] = $voce;
                    }
                }
                $liste[$campo->nome] = $items;
            }
        }

        return [$schema, $liste, $default];
    }

    public function mapCampo(Campo $campo, Permessi $permessi, array $campiEsterni = [])
    {
        $element = [
            'name' => $campo->nome,
            'type' => $campo->tipo,
            'label' => $campo->etichetta,
            'description' => $campo->descrizione,
            'rules' => [],
            'disabled' => !$permessi->canEdit($campo->permessi) || in_array($campo->nome, $campiEsterni),

        ];

        if ($campo->obbligatorio && (!in_array($permessi, [Permessi::Root, Permessi::GestioneUtentiSpeciali]))) {
            $element['rules'][] = 'required';
        }

        if ($campo->tipo === TipiCampi::Select->value) {
            $element['placeholder'] = '--';

            if ($campo->props['hidden'] ?? false) {
                if (!in_array($permessi, [Permessi::Root, Permessi::SegretarioCentrale, Permessi::SegretarioZona, Permessi::GestioneUtentiSpeciali])) {
                    $element['type'] = 'custom-hidden';
                } else {
                    $element['description'] = "CAMPO NON VISIBILE ALL'ISCRITTO\r\n".$element['description'];
                }
            }
        }

        if ($campo->tipo === TipiCampi::Text->value) {
            $element['rules'][] = 'max:250';
            if ($campo->props['email'] ?? false) {
                $element['rules'][] = 'email';
            }
            if ($campo->props['codiceFiscale'] ?? false) {
                $element['rules'][] = 'codiceFiscale';
                $element['validateOn'] = 'change';
            }
            if ($campo->props['hidden'] ?? false) {
                if (!in_array($permessi, [Permessi::Root, Permessi::SegretarioCentrale, Permessi::SegretarioZona, Permessi::GestioneUtentiSpeciali])) {
                    $element['type'] = 'custom-hidden';
                } else {
                    $element['description'] = "CAMPO NON VISIBILE ALL'ISCRITTO\r\n".$element['description'];
                }
            }
        }

        if ($campo->tipo === TipiCampi::Date->value) {

            if (true || requestIsMobile()) {
                $element['type'] = 'text';
                $element['inputType'] = 'date';
            }
            if ($campo->props['dataNascita'] ?? false) {
                if ($permessi === Permessi::Root) {
                    $element['rules'][] = 'nullable';
                }
                $element['rules'][] = 'before_or_equal:'.Carbon::now()->format('Y-m-d');
                $element['extendOptions'] = ['allowInput' => true];
            }
            if ($campo->props['maggiorenne'] ?? false) {
                $element['rules'][] = 'before_or_equal:'.Carbon::now()->subYears(18)->format('Y-m-d');
            }
        }
        if ($campo->tipo === TipiCampi::Number->value) {
            $element['type'] = 'text';
            $element['inputType'] = 'number';
            if ($permessi === Permessi::Root) {
                $element['rules'][] = 'nullable';
            }
            if (!is_null($campo->props['min'] ?? null)) {
                $element['rules'][] = 'min:'.$campo->props['min'];
            }
            if (!is_null($campo->props['max'] ?? null)) {
                $element['rules'][] = 'max:'.$campo->props['max'];
            }
            if ($campo->props['intero'] ?? false) {
                $element['rules'][] = 'integer';
            }
        }
        if ($campo->tipo === TipiCampi::Hidden->value) {
            $element['type'] = 'custom-hidden';
        }
        if ($campo->tipo === TipiCampi::Static->value) {
            if ($campo->props['nascondiTitolo'] ?? false) {
                unset($element['label']);
            }
            if (!is_null($campo->props['content'] ?? null)) {

                $element['allowHtml'] = $campo->props['html'] ?? false;

                $element['originalContent'] = $campo->props['content'];

            }

        }

        // bypass controllli nel caso di gestione utenti speciali
        if ($element['disabled'] && $permessi === Permessi::GestioneUtentiSpeciali) {
            $element['type'] = 'custom-hidden';
            $element['rules'] = [];
        }

        return $element;
    }

    public function condizioneRicorsiva($condizione, $campi)
    {
        if (!$condizione['group']) {
            // TODO: valutare se emettere un warning
            $campo = $campi[$condizione['field']] ?? null;
            if (is_null($campo)) {
                return [];
            }

            return [$campo->pagina.'.'.$campo->gruppo.'.'.$campo->nome, $condizione['operator'], $condizione['term']];
        } else {
            // TODO: valutare se emettere un warning
            return array_map(fn ($c) => $this->condizioneRicorsiva($c, $campi), $condizione['children'] ?? []);
        }
    }
}
