<?php

namespace Scribble\Data;

use Arr;
use Spatie\LaravelData\Attributes\Validation\Rule;
use Spatie\LaravelData\Data;

class CondizioneData extends Data
{
    #[Rule('required_if:group,false')]
    public null|string|int|array $term;

    #[Rule('required_if:group,true')]
    public ?TipoCondizione $type;

    #[Rule('required_if:group,false')]
    public ?string $field;

    public bool $group = false;

    /** @var array<int,CondizioneData> */
    #[Rule('required_if:group,true')]
    public ?array $children;

    #[Rule('required_if:group,false')]
    public ?string $operator;

    public static function rules()
    {
        return ['term' => 'required_if:group,false'];
    }

    public function campi()
    {
        if ($this->group) {
            return collect($this->children)->flatMap(fn ($condizione) => $condizione->campi())->unique();
        } else {
            return collect($this->field);
        }

    }

    public function verifica(array $dati, array $campi)
    {
        if ($this->group) {
            return match ($this->type) {
                TipoCondizione::And => collect($this->children)->every(fn ($condizione) => $condizione->verifica($dati, $campi)),
                TipoCondizione::Or => collect($this->children)->some(fn ($condizione) => $condizione->verifica($dati, $campi)),
            };
        } else {

            $nome = Arr::first($campi, fn ($c) => $c->id === $this->field)?->nome;
            $currentValue = $dati[$nome] ?? null;

            switch ($this->operator) {
                case '==':
                    return $currentValue == $this->term;
                case '!=':
                    return $currentValue != $this->term;
                case '>':

                    return static::nullOrEmptyCheck($this->term, $currentValue) && ($currentValue > $this->term);
                case '<':
                    return static::nullOrEmptyCheck($this->term, $currentValue) && ($currentValue < $this->term);
                case '>=':
                    return static::nullOrEmptyCheck($this->term, $currentValue) && ($currentValue >= $this->term);
                case '<=':
                    return static::nullOrEmptyCheck($this->term, $currentValue) && ($currentValue <= $this->term);
                case '*':
                    return is_string($this->term) && is_string($currentValue) && str_contains($currentValue, $this->term);
                case 'in':
                    return is_array($this->term) && in_array($currentValue, $this->term);
                default:
                    return false;
            }
        }
    }

    protected static function nullOrEmptyCheck(...$value)
    {
        foreach ($value as $v) {
            if ($v === null || $v === '') {
                return false;
            }
        }

        return true;
    }

    public static function condizioneBase()
    {
        $condizione = new CondizioneData;
        $condizione->group = true;
        $condizione->type = TipoCondizione::And;
        $condizione->children = [new CondizioneData];
        $condizione->children[0]->group = false;

        return $condizione;
    }
}

enum TipoCondizione: string
{
    case And = 'and';
    case Or = 'or';
}
