<?php

namespace Scribble;

use Cache;
use Firebase\JWT\JWT;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;

use function Illuminate\Filesystem\join_paths;

class DbManager
{
    protected Collection $useStack;

    public function __construct()
    {
        $this->useStack = collect();
    }

    protected static function stackKey($viaggioId, $version)
    {
        $version ??= 'latest';

        return "$viaggioId|||$version";
    }

    protected static function splitStackKey($key)
    {
        $parts = explode('|||', $key);

        return (object) ['viaggioId' => $parts[0], 'version' => $parts[1] === 'latest' ? null : $parts[1]];
    }

    public function dbPath($version)
    {
        return join_paths(config('scribble.storage_path'), $version.'.sqlite');
    }

    public function loadDb($viaggioId, $version = null)
    {

        if (is_null($version)) {
            $version = Cache::flexible(config('scribble.cache_key').".{$viaggioId}.version", [60, 60 * 5], function () use ($viaggioId) {
                $token = app("scribble-auth-token")->get();
                $headers = app("scribble-auth-token")->headers();
                $request = Http::withToken($token)->withHeaders($headers)->throw();
                $response = $request->get(join_paths(config('scribble.host'), "api/{$viaggioId}/version"));

                return $response->json('version');
            });
        }

        if (!file_exists($this->dbPath($version))) {
            $token = app("scribble-auth-token")->get();
            $headers = app("scribble-auth-token")->headers();
            $request = Http::withToken($token)->withHeaders($headers)->throw();
            $response = $request->get(join_paths(config('scribble.host'), "api/{$viaggioId}/db"));
            
            $newVersion = $response->getHeader('Db-Version')[0];
            if ($version != $newVersion) {
                Cache::forget(config('scribble.cache_key').".{$viaggioId}.version");
            }
            $version = $newVersion;
            file_put_contents($this->dbPath($version), $response->getBody()->getContents());
        }

        return $version;
    }

    public function useDb($viaggioId, $version = null)
    {
        $stackKey = static::stackKey($viaggioId, $version);

        if ($this->useStack->last() !== $stackKey) {
            $dbName = $this->loadDb($viaggioId, $version);
            $connectionName = config('scribble.connection.name');

            config()->set("database.connections.{$connectionName}", config('scribble.connection.template'));
            config()->set("database.connections.{$connectionName}.database", $this->dbPath($dbName));
            app(\Illuminate\Database\DatabaseManager::class)->purge($connectionName);
        }
        $this->useStack[] = $stackKey;
    }

    public function cleanUp()
    {
        $cleaningStack = $this->useStack->pop();
        if ($this->useStack->last() !== $cleaningStack) {
            $connectionName = config('scribble.connection.name');
            config()->set("database.connections.{$connectionName}", null);
            app(\Illuminate\Database\DatabaseManager::class)->purge($connectionName);
            if ($this->useStack->last()) {
                $db = static::splitStackKey($this->useStack->last());
                $this->useDb($db->viaggioId, $db->version);
            }
        }
    }
}
