<?php

namespace Scribble\Validation;

use Illuminate\Contracts\Validation\Validator;
use Illuminate\Support\MessageBag;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Validator as ValidationValidator;
use Nette\NotImplementedException;
use Scribble\Data\CampoData;
use Scribble\Data\RegolaData;
use Scribble\Data\ScribbleFormData;
use Scribble\Data\VoceData;
use Scribble\Enums\Permessi;
use Scribble\Enums\TipiCampi;
use Scribble\Rules\CondizioneFiltraRule;

class ScribbleFormValidator implements Validator
{
    protected ValidationValidator $internalValidator;

    protected $validatedRequestData;

    protected $startingRequestData;

    protected bool $run = false;

    protected $rules;

    public function __construct(protected $requestData, protected ScribbleFormData $scribbleForm, protected array $voci_terminate = [], protected Permessi $permessi = Permessi::Iscritto)
    {
        $this->startingRequestData = $this->requestData;
    }

    protected function skipField(CampoData $campo)
    {
        return $campo->tipo === TipiCampi::Static;
    }

    public function buildRules()
    {
        if (!$this->rules) {
            $campi = collect($this->scribbleForm->campi);
            $campiMap = $campi->mapWithKeys(fn ($campo) => [$campo->id => $campo->nome])->toArray();

            $internalRules = $campi->mapWithKeys(function ($campo) {
                $rules = [];

                if (!$this->skipField($campo)) {
                    if ($campo->obbligatorio && (!in_array($this->permessi, [Permessi::Root, Permessi::GestioneUtentiSpeciali]))) {
                        $rules[] = 'required';
                    } else {
                        $rules[] = 'nullable';
                    }
                    switch ($campo->tipo) {
                        case TipiCampi::Text:
                            $rules[] = 'string';
                            $rules[] = 'max:250';
                            if ($campo->props['email'] ?? false) {
                                $rules[] = 'email';
                            }
                            if ($campo->props['codiceFiscale'] ?? false) {
                                // TODO: implementare la validazione del codice fiscale
                            }
                            break;
                        case TipiCampi::TextArea:
                            $rules[] = 'string';
                            break;
                        case TipiCampi::Number:
                            $rules[] = 'numeric';
                            if ($campo->props['min'] ?? false) {
                                $rules[] = 'min:'.$campo->props['min'];
                            }
                            if ($campo->props['max'] ?? false) {
                                $rules[] = 'max:'.$campo->props['max'];
                            }
                            if ($campo->props['intero'] ?? false) {
                                $rules[] = 'integer';
                            }
                            break;
                        case TipiCampi::Date:
                            $rules[] = 'date';
                            if ($campo->props['maggiorenne'] ?? false) {
                                $rules[] = 'before_or_equal:'.now()->subYears(18)->format('Y-m-d');
                            }
                            break;
                        case TipiCampi::Checkbox:
                            $rules[] = 'boolean';
                            break;
                        case TipiCampi::Select:
                            $rules[] = function ($attribute, $value, $fail) use ($campo) {

                                if (collect($campo->voci)->filter(fn (VoceData $voce) => !in_array($voce->id, $this->voci_terminate))->search(fn (VoceData $v) => $v->valore == $value) === false) {
                                    $fail('Il campo :attribute non è valido');
                                }
                            };
                            break;
                        case TipiCampi::Hidden:
                        case TipiCampi::Static:
                        default:
                            break;
                    }
                }

                return [$campo->nome => $rules];
            })->toArray();

            $regole = collect($this->scribbleForm->regole);
            $regole->each(function (RegolaData $regola) use (&$internalRules, $campiMap) {
                $condizioneKey = '__condizione_'.$regola->id;
                foreach ($regola->mostra_campi as $campo) {
                    $key = $campiMap[$campo];
                    $currentRules = $internalRules[$key];
                    $internalRules[$key] = ["exclude_unless:$condizioneKey,true", ...$currentRules];
                }
                foreach ($regola->filtra_voci as $filtroVoci) {
                    $key = $campiMap[$filtroVoci->campo];
                    $currentRules = $internalRules[$key];
                    $internalRules[$key][] = new CondizioneFiltraRule($condizioneKey, $filtroVoci);
                }
            });
            $this->rules = $internalRules;
        }

        return $this->rules;
    }

    public function conditionalData($formData)
    {
        return collect($this->scribbleForm->regole)
            ->mapWithKeys(fn (RegolaData $regola) => ['__condizione_'.$regola->id => $regola->condizione->verifica($formData, $this->scribbleForm->campi)])->toArray();
    }

    public function validate()
    {
        if (!$this->passes()) {
            throw new ValidationException($this->internalValidator);
        }

        return $this->validated();
    }

    protected function stepValidate()
    {
        $starting = $this->requestData;
        $merged = array_merge($this->requestData, $this->conditionalData($starting));
        $this->internalValidator = \Validator::make($merged, $this->buildRules());

        $this->requestData = $this->internalValidator->validated();

        return json_encode($this->requestData) !== json_encode($starting);
    }

    public function fails()
    {
        if (!$this->run) {
            $this->passes();
        }

        return $this->internalValidator->errors()->count() > 0;
    }

    public function failed() {}

    public function sometimes($attribute, $rules, callable $callback)
    {
        throw new NotImplementedException;
    }

    public function after($callback)
    {
        throw new NotImplementedException;
    }

    public function errors()
    {
        return $this->getMessageBag();
    }

    public function getMessageBag()
    {
        if (!$this->run) {
            $this->passes();
        }

        return $this->internalValidator->errors();
    }

    protected static function remapPrefixes(MessageBag $errors, $prefix)
    {
        $remapped = [];
        foreach ($errors->messages() as $key => $messages) {
            $prefixedKey = $prefix.'.'.$key;
            if ($prefix === '') {
                $prefixedKey = $key;
            }
            $remapped[$prefixedKey] = array_map(fn ($el) => str_replace($key, $prefixedKey, $el), $messages);
        }

        return $remapped;
    }

    public function passes()
    {
        $this->run = true;
        try {
            while ($this->stepValidate()) {

            }
        } catch (ValidationException $e) {
            // \Log::error(__METHOD__, ['eccezione' => $e, 'contesto' => context_error_details(false)]);

            return false;
        }

        $this->validatedRequestData = $this->requestData;

        return $this->internalValidator->errors()->count() === 0;
    }

    public function validated()
    {
        if (!$this->run) {
            $this->validate();
        }

        return $this->validatedRequestData;
    }
}
