<?php

/**
 * Created by Reliese Model.
 */

namespace Txd\Tags\Traits;

use ArrayAccess;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Support\Collection;
use Txd\Tags\Classes\TagValue;
use Txd\Tags\Models\Tag;
use Txd\Tags\Models\Taggable;

/**
 * @property Collection<int, TagValue> $queueTags
 */
trait HasTags
{
    protected ?Collection $queuedTags = null;

    public static function getTagClassName(): string
    {
        return config('tags.tag_model', Tag::class);
    }

    public static function getTagTableName(): string
    {
        $tagInstance = new (self::getTagClassName());

        return $tagInstance->getTable();
    }

    public static function getTaggableMorphName(): string
    {
        return config('tags.taggable.morph_name', 'taggable');
    }

    public static function getTaggableTableName(): string
    {
        return config('tags.taggable.table_name', 'taggables');
    }

    public static function getPivotModelClassName(): string
    {
        return config('tags.taggable.class_name', Taggable::class);
    }

    public function tags(): MorphToMany
    {
        return $this
            ->morphToMany(self::getTagClassName(), $this->getTaggableMorphName(), $this->getTaggableTableName())
            ->using($this->getPivotModelClassName())->withPivot(['value', 'forced']);
    }

    public function setTagsAttribute(string|array|ArrayAccess|Tag $tags)
    {
        $this->queuedTags = static::convertToTags($tags);
        static::saved(function (Model $taggableModel) {
            $taggableModel->syncTags();
        });

    }

    public static function bootHasTags()
    {
        static::created(function (Model $taggableModel) {
            if (isset($taggableModel->queuedTags) && count($taggableModel->queuedTags) === 0) {
                return;
            }
            $taggableModel->syncTags();
        });

        static::deleted(function (Model $deletedModel) {
            $tags = $deletedModel->tags()->get();

            $deletedModel->detachTags($tags);
        });
    }

    public function attachTags(array|ArrayAccess|Tag $tags): static
    {
        $tags = static::convertToTags($tags);

        $this->tags()->syncWithoutDetaching($tags->mapWithKeys(function (TagValue $tagValue) {
            return [$tagValue->tag->getKey() => ['value' => $tagValue->value]];
        }));

        return $this;
    }

    public function attachTag(string|Tag $tag)
    {
        return $this->attachTags([$tag]);
    }

    public function detachTags(array|ArrayAccess $tags): static
    {
        $tags = static::convertToTags($tags);

        collect($tags)
            ->filter()
            ->each(fn (TagValue $tag) => $this->tags()->detach($tag->tag->getKey()));

        return $this;
    }

    public function detachTag(string|Tag $tag): static
    {
        return $this->detachTags([$tag]);
    }

    public function syncTags(): static
    {
        if (!isset($this->queuedTags)) {
            return $this;
        }
        $tags = $this->queuedTags->mapWithKeys(function (TagValue $tagValue) {

            return [$tagValue->tag->getKey() => ['value' => trim($tagValue->value)]];
        });
        $this->tags()->sync($tags);

        return $this;
    }

    /**
     * @return Collection<string|int, TagValue>
     */
    public static function convertToTags(string|array|ArrayAccess|Tag $tags)
    {
        $out = collect();

        if (is_iterable($tags)) {
            foreach ($tags as $tag) {
                if (is_a($tag, static::getTagClassName())) {
                    $out->push(TagValue::fromTag($tag));
                } elseif (is_string($tag)) {
                    $out->push(TagValue::fromString($tag));
                } else {
                    throw new Exception('Bad Value for Tag');
                }
            }
        } else {
            if (is_a($tags, static::getTagClassName())) {
                $out->push(TagValue::fromTag($tags));
            } elseif (is_string($tags)) {
                $out->push(TagValue::fromString($tags));
            } else {
                throw new Exception('Bad Value for Tag');
            }
        }

        return $out;
    }

    public function scopeWhereTags($query, $value)
    {
        if (empty($value)) {
            return $query;
        }
        if (!is_array($value)) {
            $value = [$value];
        }

        $value = array_map(fn ($el) => explode(':', $el), $value);

        $query->whereHas('tags', fn ($q) => $q->where('taggables.forced', '>=', 0)->where(function ($q) use ($value) {
            foreach ($value as $t) {
                $q->orWhere(fn ($q) => $q->where('tags.name', $t[0])->when($t[1] ?? false, fn ($q) => $q->where('taggables.value', $t[1])));
            }

            return $q;
        }));

        return $query;
    }
}
