<?php

namespace Txd\Translations;

use Illuminate\Support\Facades\Cache as FacadesCache;

class TranslationServiceProvider extends \Illuminate\Translation\TranslationServiceProvider
{
    public function boot()
    {
        $this->publishes([
            __DIR__.'/../config/txd-translations.php' => config_path('txd-translations.php'),
        ], 'txd-translations:config');

        $this->mergeConfigFrom(
            __DIR__.'/../config/txd-translations.php', 'txd-translations'
        );

        if (config('txd-translations.automatic_generation_from_db', true)) {
            \Barryvdh\TranslationManager\Models\Translation::saved(function () {
                $command = new \Txd\Translations\Console\RenewTranslationsCacheCommand;
                $command->handle();
            });
        }
    }

    protected function registerTranslator()
    {
        $this->app->singleton('translator', function ($app) {
            $loader = $app['translation.loader'];

            // When registering the translator component, we'll need to set the default
            // locale as well as the fallback locale. So, we'll grab the application
            // configuration so we can easily get both of these values from there.
            $locale = $app['config']['app.locale'];

            $trans = new Translator($loader, $locale);

            $trans->setFallback($app['config']['app.fallback_locale']);
            $trans->determineLocalesUsing(function () use ($trans) {
                return array_filter([$trans->locale(), $trans->getFallback(), 'default']);
            });

            if ($app->bound('translation-manager')) {
                $trans->setTranslationManager($app['translation-manager']);
            }

            return $trans;
        });
    }

    protected function registerCommand()
    {
        $this->app->singleton('command.txd-translations.cache', function ($app) {
            return new Console\RenewTranslationsCacheCommand;
        });
        $this->commands('command.txd-translations.cache');
    }

    public function register()
    {
        $this->app->singleton('command.translation-manager.import', function () {
            return new Console\ImportDisabledCommand;
        });

        $this->registerCommand();

        $this->app->resolving('translation.loader', function ($instance, $app) {
            if (config('txd-translations.automatic_generation_from_db', true) && !$app->runningInConsole()) {
                $value = cache('ltm_translations_version');
                $db_value = txdTranslationsVersion();

                if (empty($value) || empty($db_value) || $value !== $db_value) {
                    app(\Barryvdh\TranslationManager\Manager::class)->exportAllTranslations();
                    FacadesCache::forever('ltm_translations_version', $db_value);
                }
            }
        });
        $this->app->resolving('translator', function ($trans, $app) {
            $trans->determineLocalesUsing(function () use ($trans) {
                return array_filter([$trans->locale(), $trans->getFallback(), 'default']);
            });
        });
        if ($this->app['config']['txd-translations.listen'] ?? false) {
            $this->registerLoader();
            $this->registerTranslator();
        } else {
            parent::register();
        }

    }
}
